<?php
// FILE: admin/membership.php
// PURPOSE: To view, search, and manage Genowa Movement members.

require_once 'functions.php';
require_once 'db.php';

// Security: Only SuperAdmins and Admins can access this page
require_login();
if (!is_super_admin() && !is_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- Setup Search and Pagination ---
$search_term = trim($_GET['search'] ?? '');
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 15;
$offset = ($page - 1) * $limit;

$members = [];
$total_members = 0;

try {
    // --- Build Dynamic SQL Query ---
    $sql_base = "FROM movement_members m JOIN wards w ON m.ward_id = w.id";
    $where_clauses = [];
    $params = [];

    if (!empty($search_term)) {
        $where_clauses[] = "(m.name LIKE :search OR m.email LIKE :search OR m.phone_number LIKE :search OR m.membership_id = :search_exact)";
        $params['search'] = '%' . $search_term . '%';
        $params['search_exact'] = $search_term; // For exact match on membership ID
    }

    $sql_where = !empty($where_clauses) ? " WHERE " . implode(" AND ", $where_clauses) : "";

    // Get total count for pagination
    $count_sql = "SELECT COUNT(m.id) " . $sql_base . $sql_where;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_members = $count_stmt->fetchColumn();
    $total_pages = ceil($total_members / $limit);

    // Get members for the current page
    $members_sql = "SELECT m.*, w.name as ward_name " . $sql_base . $sql_where . " ORDER BY m.joined_at DESC LIMIT :limit OFFSET :offset";
    $members_stmt = $pdo->prepare($members_sql);
    
    // Bind search params if they exist
    foreach ($params as $key => &$val) {
        $members_stmt->bindParam($key, $val);
    }
    
    $members_stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $members_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    
    $members_stmt->execute();
    $members = $members_stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching members: " . $e->getMessage();
}


include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">Genowa Movement Membership</h1>
        <p class="mt-1 text-sm text-gray-500">Total members: <?php echo e(number_format($total_members)); ?></p>
    </div>
</header>
    
<div class="bg-white p-4 my-6 rounded-lg shadow">
    <form action="membership.php" method="get">
        <label for="search" class="sr-only">Search</label>
        <div class="flex rounded-md shadow-sm">
            <div class="relative flex-grow focus-within:z-10">
                <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" viewBox="0 0 20 20" fill="currentColor"><path fill-rule="evenodd" d="M9 3.5a5.5 5.5 0 100 11 5.5 5.5 0 000-11zM2 9a7 7 0 1112.452 4.391l3.328 3.329a.75.75 0 11-1.06 1.06l-3.329-3.328A7 7 0 012 9z" clip-rule="evenodd" /></svg></div>
                <input type="search" name="search" id="search" value="<?php echo e($search_term); ?>" class="block w-full rounded-none rounded-l-md border-0 py-2.5 pl-10 text-gray-900 ring-1 ring-inset ring-gray-300 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-brand-purple sm:text-sm" placeholder="Search by name, email, phone, or ID">
            </div>
            <button type="submit" class="relative -ml-px inline-flex items-center gap-x-1.5 rounded-r-md px-3 py-2 text-sm font-semibold text-gray-900 ring-1 ring-inset ring-gray-300 hover:bg-gray-50">Search</button>
        </div>
    </form>
</div>

<div class="overflow-x-auto bg-white rounded-lg shadow">
    <?php if (isset($db_error)): ?>
        <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
    <?php elseif (empty($members)): ?>
        <div class="text-center p-12">
            <h3 class="mt-2 text-lg font-medium text-gray-900">No members found.</h3>
            <?php if($search_term): ?><p class="mt-1 text-sm text-gray-500">Try adjusting your search term.</p><?php endif; ?>
        </div>
    <?php else: ?>
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Membership ID</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Contact</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ward</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date Joined</th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                <?php foreach ($members as $member): ?>
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-semibold text-brand-purple"><?php echo e($member['membership_id']); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900"><?php echo e($member['name']); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm text-gray-900"><?php echo e($member['email']); ?></div>
                            <div class="text-sm text-gray-500"><?php echo e($member['phone_number']); ?></div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo e($member['ward_name']); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date('M j, Y', strtotime($member['joined_at'])); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?search=<?php echo urlencode($search_term); ?>&page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    
    <div class="text-sm text-gray-700">
        Page <?php echo $page; ?> of <?php echo $total_pages; ?>
    </div>

    <?php if ($page < $total_pages): ?>
        <a href="?search=<?php echo urlencode($search_term); ?>&page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
