<?php
// FILE: admin/functions.php
// PURPOSE: To store reusable functions for the admin panel.

// SETS THE DEFAULT TIMEZONE FOR THE ENTIRE APPLICATION
date_default_timezone_set('Africa/Nairobi');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * NEW: Checks for pending items and notifies SuperAdmins if thresholds are met.
 * Includes a cooldown period to prevent spamming.
 * @param PDO $pdo The database connection object.
 * @return void
 */
function check_and_notify_pending_items(PDO $pdo): void
{
    $notification_cooldown = 3600 * 6; // 6 hours in seconds
    $notification_dir = __DIR__ . '/notifications/';
    if (!is_dir($notification_dir)) {
        mkdir($notification_dir, 0755, true);
    }

    $last_project_notification_file = $notification_dir . 'last_project_notification.txt';
    $last_user_notification_file = $notification_dir . 'last_user_notification.txt';

    try {
        // 1. Check pending projects
        $project_count_stmt = $pdo->query("SELECT COUNT(id) FROM projects WHERE is_approved = 0");
        $pending_projects_count = $project_count_stmt->fetchColumn();

        if ($pending_projects_count > 10) {
            $last_sent_time = file_exists($last_project_notification_file) ? filemtime($last_project_notification_file) : 0;
            if (time() - $last_sent_time > $notification_cooldown) {
                // Fetch SuperAdmins
                $super_admins_stmt = $pdo->query("SELECT email, phone_number FROM users WHERE role = 'SuperAdmin' AND is_active = 1");
                $super_admins = $super_admins_stmt->fetchAll();

                $message = "Genowa Alert: There are " . $pending_projects_count . " projects pending approval. Please log in to the admin panel to review them.";
                $subject = "Alert: High Number of Pending Projects";

                foreach ($super_admins as $admin) {
                    if (!empty($admin['email'])) {
                        send_email($admin['email'], $subject, "<p>" . $message . "</p>");
                    }
                    if (!empty($admin['phone_number'])) {
                        send_sms($admin['phone_number'], $message);
                    }
                }
                touch($last_project_notification_file); // Update the file's timestamp
            }
        }

        // 2. Check pending users
        $user_count_stmt = $pdo->query("SELECT COUNT(id) FROM users WHERE is_active = 0");
        $pending_users_count = $user_count_stmt->fetchColumn();

        if ($pending_users_count > 3) {
            $last_sent_time = file_exists($last_user_notification_file) ? filemtime($last_user_notification_file) : 0;
             if (time() - $last_sent_time > $notification_cooldown) {
                // Fetch SuperAdmins
                $super_admins_stmt = $pdo->query("SELECT email, phone_number FROM users WHERE role = 'SuperAdmin' AND is_active = 1");
                $super_admins = $super_admins_stmt->fetchAll();
                
                $message = "Genowa Alert: There are " . $pending_users_count . " new users awaiting approval. Please log in to the User Management page to review.";
                $subject = "Alert: New Users Awaiting Approval";

                foreach ($super_admins as $admin) {
                    if (!empty($admin['email'])) {
                        send_email($admin['email'], $subject, "<p>" . $message . "</p>");
                    }
                    if (!empty($admin['phone_number'])) {
                        send_sms($admin['phone_number'], $message);
                    }
                }
                touch($last_user_notification_file); // Update the file's timestamp
             }
        }

    } catch (PDOException $e) {
        // Silently log error, don't break the page
        error_log("Notification Check Error: " . $e->getMessage());
    }
}


/**
 * Sends an email.
 * @param string $to The recipient's email address.
 * @param string $subject The subject of the email.
 * @param string $body The HTML body of the email.
 * @return bool True on success, false on failure.
 */
function send_email(string $to, string $subject, string $body): bool
{
    $headers = "MIME-Version: 1.0" . "\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $headers .= 'From: Genowa Delivery Unit <hello@genowa.co.ke>' . "\r\n";
    return mail($to, $subject, $body, $headers);
}

/**
 * Sends an SMS using the Onfon Media API.
 * @param string $phoneNumber The recipient's phone number.
 * @param string $message The message to send.
 * @return bool True on success, false on failure.
 */
function send_sms(string $phoneNumber, string $message): bool
{
    // Sanitize phone number
    if (substr($phoneNumber, 0, 1) == '+') { $phoneNumber = substr($phoneNumber, 1); }
    if (substr($phoneNumber, 0, 1) == '0') { $phoneNumber = '254' . substr($phoneNumber, 1); }
    if (substr($phoneNumber, 0, 3) != '254') { return false; }

    $api_url = 'https://api.onfonmedia.co.ke/v1/sms/SendBulkSMS';
    $api_key = 'HDv7xtZaIBLJwVlbkfYj65z2pRQGsOoPqihgy8n140AF39Km';
    $client_id = 'Agwesh';
    $sender_id = 'GENOWA';

    $payload = [
        'SenderId' => $sender_id,
        'MessageParameters' => [['Number' => $phoneNumber, 'Text' => $message]],
        'ApiKey' => $api_key,
        'ClientId' => $client_id
    ];

    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json', 'AccessKey: ' . $api_key]);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code == 200) {
        $result = json_decode($response, true);
        return isset($result['Data'][0]['ResultCode']) && $result['Data'][0]['ResultCode'] == 0;
    }
    return false;
}


// --- All other existing functions ---
function require_login(): void { if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) { header("location: login.php"); exit; } }
function is_super_admin(): bool { return isset($_SESSION["role"]) && $_SESSION["role"] === 'SuperAdmin'; }
function is_admin(): bool { return isset($_SESSION["role"]) && $_SESSION["role"] === 'Admin'; }
function is_approver(): bool { return isset($_SESSION["role"]) && $_SESSION["role"] === 'Approver'; }
function is_inputer(): bool { return isset($_SESSION["role"]) && $_SESSION["role"] === 'Inputer'; }
function require_super_admin(): void { if (!is_super_admin()) { header("location: dashboard.php"); exit; } }
function log_action(PDO $pdo, string $action, string $target_type, int $target_id): void {
    $sql = "INSERT INTO audit_trails (user_id, user_name, action, target_type, target_id) VALUES (:user_id, :user_name, :action, :target_type, :target_id)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(['user_id' => $_SESSION['id'] ?? 0, 'user_name' => $_SESSION['name'] ?? 'System', 'action' => $action, 'target_type' => $target_type, 'target_id' => $target_id]);
}
function e(?string $string): string { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }
