<?php
// SETS THE DEFAULT TIMEZONE FOR THE ENTIRE APPLICATION
date_default_timezone_set('Africa/Nairobi');

require_once 'functions.php';
require_once 'db.php';

// Security: Only SuperAdmins can access this page
require_login();
require_super_admin();

// --- Pagination Setup ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20; // Number of log entries per page
$offset = ($page - 1) * $limit;

$audit_logs = [];
$total_logs = 0;

try {
    // Get total count for pagination
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM audit_trails");
    $total_logs = $count_stmt->fetchColumn();
    $total_pages = ceil($total_logs / $limit);

    // Fetch the logs for the current page
    $log_stmt = $pdo->prepare("SELECT * FROM audit_trails ORDER BY timestamp DESC LIMIT :limit OFFSET :offset");
    $log_stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $log_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $log_stmt->execute();
    $audit_logs = $log_stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching audit trails: " . $e->getMessage();
}


include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">System Audit Trails</h1>
        <p class="mt-2 text-sm text-gray-700">A record of all significant actions performed by users in the system.</p>
    </div>
</header>
    
<!-- Main content starts here -->
<div class="overflow-x-auto bg-white rounded-lg shadow mt-6">
    <?php if (isset($db_error)): ?>
        <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
    <?php elseif (empty($audit_logs)): ?>
        <div class="text-center p-12">
            <h3 class="mt-2 text-lg font-medium text-gray-900">No system actions have been logged yet.</h3>
        </div>
    <?php else: ?>
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timestamp</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Target</th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                <?php foreach ($audit_logs as $log): ?>
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm text-gray-900"><?php echo date('M j, Y, g:i a', strtotime($log['timestamp'])); ?></div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900"><?php echo e($log['user_name']); ?></div>
                            <div class="text-sm text-gray-500">ID: <?php echo e($log['user_id']); ?></div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                             <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                <?php echo e($log['action']); ?>
                            </span>
                        </td>
                         <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm text-gray-900"><?php echo e($log['target_type']); ?> #<?php echo e($log['target_id']); ?></div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    
    <div class="text-sm text-gray-700">
        Page <?php echo $page; ?> of <?php echo $total_pages; ?>
    </div>

    <?php if ($page < $total_pages): ?>
        <a href="?page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<?php 
// The footer now closes the main content area and the page
include 'includes/footer.php'; 
?>
