<?php
// FILE: public_html/submit_feedback.php
// PURPOSE: Handles AJAX submission of public feedback for a project.

header('Content-Type: application/json');

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

require_once 'db_connect.php';
require_once 'admin/functions.php'; // For e()

$response = ['success' => false, 'message' => 'An unknown error occurred.'];

// --- Basic Request Validation ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = 'Invalid request method.';
    echo json_encode($response);
    exit;
}

// --- CSRF Token Validation ---
if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $response['message'] = 'Invalid or expired form submission. Please refresh the page and try again.';
    echo json_encode($response);
    exit;
}

// --- Input Sanitization and Validation ---
$project_id = filter_input(INPUT_POST, 'project_id', FILTER_VALIDATE_INT);
$author_name = trim($_POST['author_name'] ?? '');
$comment = trim($_POST['comment'] ?? '');
$errors = [];

if (!$project_id) {
    $errors[] = 'Invalid project reference.';
}
if (empty($author_name)) {
    $errors[] = 'Your name is required.';
}
if (empty($comment)) {
    $errors[] = 'A comment is required.';
}
if (strlen($author_name) > 100) {
    $errors[] = 'Name must be 100 characters or less.';
}
if (strlen($comment) > 2000) {
    $errors[] = 'Comment must be 2000 characters or less.';
}

if (!empty($errors)) {
    $response['message'] = implode(' ', $errors);
    echo json_encode($response);
    exit;
}

// --- Database Insertion ---
try {
    // Verify project exists and is approved
    $check_stmt = $pdo->prepare("SELECT id FROM projects WHERE id = :id AND is_approved = 1");
    $check_stmt->execute(['id' => $project_id]);
    if (!$check_stmt->fetch()) {
        $response['message'] = 'This project does not exist or is not currently accepting feedback.';
        echo json_encode($response);
        exit;
    }

    // Insert the feedback, it will be unapproved by default
    $sql = "INSERT INTO project_feedback (project_id, author_name, comment, is_approved) VALUES (:project_id, :author_name, :comment, 0)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'project_id' => $project_id,
        'author_name' => $author_name,
        'comment' => $comment
    ]);

    $response['success'] = true;
    $response['message'] = 'Thank you for your feedback! Your comment has been submitted and is awaiting approval from an administrator.';

} catch (PDOException $e) {
    error_log("Feedback Submission PDOException: " . $e->getMessage());
    $response['message'] = 'A database error occurred. Please try again later.';
}

echo json_encode($response);
?>
