<?php
// FILE: public_submit_article.php
// --- MODIFIED ---: Enhanced to new standards with modern UI, rich text editor, categories, and tags.

require_once 'admin/functions.php';
require_once 'admin/db.php';
session_start();

// --- NEW: Define the ID for the default guest author ---
const GUEST_AUTHOR_ID = 0;

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Initialize form variables
$article = [
    'title' => '', 'content' => '', 'project_id' => null, 'category_id' => null,
    'author_name' => '', 'email' => ''
];
$selected_tags = [];
$errors = [];
$success = false;

// --- MODIFIED: Fetch all data needed for form dropdowns ---
try {
    $approved_projects = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC")->fetchAll();
    $categories = $pdo->query("SELECT id, name FROM categories ORDER BY name ASC")->fetchAll();
    $tags = $pdo->query("SELECT id, name FROM tags ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db'] = "Could not load required page data.";
}

// Handle form submit
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    // CSRF check
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $errors['csrf'] = 'Invalid form submission. Please refresh and try again.';
    } else {
        $article['title'] = trim($_POST['title']);
        $article['content'] = trim($_POST['content']);
        $article['project_id'] = empty($_POST['project_id']) ? null : $_POST['project_id'];
        $article['category_id'] = empty($_POST['category_id']) ? null : $_POST['category_id'];
        $article['author_name'] = trim($_POST['author_name']);
        $article['email'] = trim($_POST['email']);
        $selected_tags = $_POST['tags'] ?? [];

        // Validation
        if (empty($article['title'])) $errors['title'] = "Title is required.";
        if (empty($article['content']) || strlen(strip_tags($article['content'])) < 50) $errors['content'] = "Content must be at least 50 characters.";
        if (empty($article['project_id'])) $errors['project_id'] = "Please select a related project.";
        if (empty($article['category_id'])) $errors['category_id'] = "Please select a category.";
        if (empty($article['author_name'])) $errors['author_name'] = "Your name is required.";
        if (!filter_var($article['email'], FILTER_VALIDATE_EMAIL)) $errors['email'] = "A valid email address is required.";

        // --- MODIFIED: Slug generation to ensure uniqueness ---
        $slug = create_slug($article['title']) . '-' . bin2hex(random_bytes(3));
        
        // Image upload (optional)
        $new_image_filename = null;
        if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
            // ... (your existing strict file validation logic here) ...
            $ext = pathinfo($_FILES['featured_image']['name'], PATHINFO_EXTENSION);
            $filename = 'public_' . time() . '.' . $ext;
            if (move_uploaded_file($_FILES['featured_image']['tmp_name'], '../admin/uploads/' . $filename)) {
                $new_image_filename = $filename;
            }
        }

        // Save article if no errors
        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                // --- MODIFIED: INSERT statement now includes author_id and category_id ---
                $stmt = $pdo->prepare("
                    INSERT INTO news_articles 
                    (title, slug, content, status, project_id, category_id, featured_image, author_id, author_name, email) 
                    VALUES 
                    (:title, :slug, :content, 'Draft', :project_id, :category_id, :featured_image, :author_id, :author_name, :email)
                ");
                $stmt->execute([
                    'title' => $article['title'],
                    'slug' => $slug,
                    'content' => $article['content'],
                    'project_id' => $article['project_id'],
                    'category_id' => $article['category_id'],
                    'featured_image' => $new_image_filename,
                    'author_id' => GUEST_AUTHOR_ID, // Use the guest author ID
                    'author_name' => $article['author_name'],
                    'email' => $article['email']
                ]);
                $article_id = $pdo->lastInsertId();

                // --- NEW: Insert selected tags ---
                if (!empty($selected_tags) && $article_id) {
                    $tag_stmt = $pdo->prepare("INSERT INTO article_tags (article_id, tag_id) VALUES (:article_id, :tag_id)");
                    foreach ($selected_tags as $tag_id) {
                        $tag_stmt->execute(['article_id' => $article_id, 'tag_id' => (int)$tag_id]);
                    }
                }

                $pdo->commit();
                $success = true;

            } catch (PDOException $e) {
                $pdo->rollBack();
                error_log('Public submission error: ' . $e->getMessage());
                $errors['db'] = "Could not submit your article due to a technical error. Please try again later.";
            }
        }
    }
}

// Use a generic header for public pages, or the admin one if appropriate
include 'includes/header.php'; // Assuming a public header exists
?>

<link href="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/css/tom-select.bootstrap5.min.css" rel="stylesheet">
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<style>
    .ts-control { 
        padding: 0.6rem 0.75rem; 
    }
    .ql-editor { 
        min-height: 200px; 
    }
    .form-label { 
        display: block; 
        margin-bottom: 0.5rem; 
        font-weight: 500; 
        color: #374151; 
    }
    .form-input, .ts-wrapper { 
        display: block; 
        width: 100%; 
        border-radius: 0.5rem; 
        border: 1px solid #d1d5db; 
        padding: 0.5rem 0.75rem; 
    }
    .form-error { 
        color: #dc2626; 
        font-size: 0.875rem; 
        margin-top: 0.25rem; 
    }

    /* --- NEW: Style for Tom Select Dropdown Panel --- */
    .ts-dropdown {
        background-color: #fff; /* Solid white background */
        z-index: 20; /* Ensure it's above other page content */
        border: 1px solid #d1d5db; /* Consistent border */
        border-radius: 0.5rem; /* Consistent corner radius */
        box-shadow: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    }
</style>

<main class="bg-gray-50 py-12">
    <div class="max-w-4xl mx-auto p-8 bg-white rounded-2xl shadow-lg">
        <h1 class="text-3xl font-bold text-gray-800 mb-2">Submit Your Story</h1>
        <p class="text-gray-600 mb-8">Share an update on a project or an event happening in your community. All submissions are reviewed before publishing.</p>

        <?php if ($success): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-6 rounded-lg">
                <h3 class="font-bold text-lg">✅ Thank You!</h3>
                <p>Your submission has been received and is pending moderation. We appreciate your contribution.</p>
            </div>
        <?php else: ?>
            <?php if (!empty($errors)): ?>
                <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="space-y-8">
                <input type="hidden" name="csrf_token" value="<?php echo e($_SESSION['csrf_token']); ?>">
                <input type="hidden" name="content" id="quill-content"> <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                    <div>
                        <label for="author_name" class="form-label">Your Name</label>
                        <input type="text" name="author_name" id="author_name" class="form-input" value="<?php echo e($article['author_name']); ?>" required>
                    </div>
                    <div>
                        <label for="email" class="form-label">Email Address</label>
                        <input type="email" name="email" id="email" class="form-input" value="<?php echo e($article['email']); ?>" required>
                    </div>
                </div>

                <div>
                    <label for="title" class="form-label">Article Title</label>
                    <input type="text" name="title" id="title" class="form-input" value="<?php echo e($article['title']); ?>" required>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                    <div>
                        <label for="project_id" class="form-label">Related Project</label>
                        <select name="project_id" id="project_id" required>
                            <option value="">Select a project...</option>
                            <?php foreach ($approved_projects as $proj): ?>
                                <option value="<?php echo $proj['id']; ?>" <?php if ($article['project_id'] == $proj['id']) echo 'selected'; ?>>
                                    <?php echo e($proj['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="category_id" class="form-label">Category</label>
                        <select name="category_id" id="category_id" required>
                             <option value="">Select a category...</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>" <?php if ($article['category_id'] == $cat['id']) echo 'selected'; ?>>
                                    <?php echo e($cat['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div>
                    <label for="content" class="form-label">Article Content</label>
                    <div id="editor" class="bg-white rounded-lg border"><?php echo $article['content']; ?></div>
                </div>

                 <div>
                    <label for="tags" class="form-label">Tags (select up to 5)</label>
                    <select name="tags[]" id="tags" multiple>
                        <?php foreach ($tags as $tag): ?>
                            <option value="<?php echo $tag['id']; ?>" <?php if (in_array($tag['id'], $selected_tags)) echo 'selected'; ?>>
                                <?php echo e($tag['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label for="featured_image" class="form-label">Featured Image (optional)</label>
                    <input type="file" name="featured_image" id="featured_image" accept="image/*" class="form-input">
                </div>

                <div class="pt-4 border-t">
                    <button type="submit" class="w-full md:w-auto bg-brand-blue text-white px-8 py-3 rounded-lg font-semibold text-lg hover:bg-blue-800 transition-colors">
                        Submit for Review
                    </button>
                </div>
            </form>
        <?php endif; ?>
    </div>
</main>

<script src="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/js/tom-select.complete.min.js"></script>
<script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Quill Editor
    var quill = new Quill('#editor', {
        theme: 'snow',
        modules: {
            toolbar: [['bold', 'italic'], ['link', 'blockquote'], [{ 'list': 'ordered'}, { 'list': 'bullet' }]]
        }
    });

    // Tom Select
    new TomSelect('#project_id');
    new TomSelect('#category_id');
    new TomSelect('#tags', { maxItems: 5, plugins: ['remove_button'] });

    // Link Quill to hidden input on form submit
    const form = document.querySelector('form');
    if(form) {
        form.addEventListener('submit', function() {
            document.getElementById('quill-content').value = quill.root.innerHTML;
        });
    }
});
</script>

<?php include 'includes/footer.php';  ?>