<?php 
// FILE: public_html/projects.php
// PURPOSE: Displays a comprehensive, filterable list of all projects.

require_once 'db_connect.php';
require_once 'admin/functions.php'; // For e() and generate_pagination() helpers

// --- Initial Setup ---
$filters = [
    'search' => trim($_GET['search'] ?? ''),
    'constituency' => trim($_GET['constituency'] ?? ''),
    'ward' => trim($_GET['ward'] ?? ''),
    'status' => $_GET['status'] ?? [],
    'pillar' => trim($_GET['pillar'] ?? '')
];
if (!is_array($filters['status'])) { $filters['status'] = []; }
$filters['status'] = array_map('htmlspecialchars', $filters['status']);

$page = isset($_GET['page']) && (int)$_GET['page'] > 0 ? (int)$_GET['page'] : 1;
$limit = 12;
$offset = ($page - 1) * $limit;

$projects = $constituencies = $all_wards = $pillars = [];
$total_projects = 0;

$sql_base = "FROM projects p 
             JOIN wards w ON p.ward_id = w.id 
             JOIN constituencies c ON w.constituency_id = c.id";
$where_clauses = [];
$params = [];

if (!empty($filters['search'])) {
    $where_clauses[] = "(p.name LIKE :search OR p.description LIKE :search)";
    $params['search'] = '%' . $filters['search'] . '%';
}
if (!empty($filters['constituency'])) {
    $where_clauses[] = "c.id = :constituency";
    $params['constituency'] = $filters['constituency'];
}
if (!empty($filters['ward'])) {
    $where_clauses[] = "w.id = :ward";
    $params['ward'] = $filters['ward'];
}
if (!empty($filters['status'])) {
    $status_placeholders = [];
    foreach ($filters['status'] as $key => $value) {
        $placeholder = ':status' . $key;
        $status_placeholders[] = $placeholder;
        $params[$placeholder] = $value;
    }
    $where_clauses[] = "p.status IN (" . implode(', ', $status_placeholders) . ")";
}
if (!empty($filters['pillar'])) {
    $where_clauses[] = "p.manifesto_pillar_id = :pillar";
    $params['pillar'] = $filters['pillar'];
}
$where_clauses[] = "p.is_approved = 1";
$sql_where = " WHERE " . implode(" AND ", $where_clauses);

try {
    $count_sql = "SELECT COUNT(p.id) " . $sql_base . $sql_where;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_projects = $count_stmt->fetchColumn();
    $total_pages = $total_projects > 0 ? ceil($total_projects / $limit) : 1;

    if ($page > $total_pages) { $page = $total_pages; }
    if ($page < 1) { $page = 1; }
    $offset = ($page - 1) * $limit;

    $projects_sql = "SELECT p.id, p.name, p.status, p.description, w.name as ward_name, c.name as constituency_name, pi.image_url as project_image
                     " . $sql_base . "
                     LEFT JOIN (
                         SELECT project_id, image_url, ROW_NUMBER() OVER(PARTITION BY project_id ORDER BY photo_type='After' DESC, id DESC) as rn
                         FROM project_photos
                     ) pi ON p.id = pi.project_id AND pi.rn = 1
                     " . $sql_where . " ORDER BY p.updated_at DESC LIMIT :limit OFFSET :offset";

    $projects_stmt = $pdo->prepare($projects_sql);
    foreach ($params as $key => &$val) { $projects_stmt->bindParam($key, $val); }
    $projects_stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $projects_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $projects_stmt->execute();
    $projects = $projects_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $constituencies = $pdo->query("SELECT id, name FROM constituencies ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
    $all_wards = $pdo->query("SELECT id, name, constituency_id FROM wards ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
    $pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $db_error_message = "A database error occurred. Please try again later.";
    error_log("All Projects Page DB Error: " . $e->getMessage());
}

function highlight_term($text, $term) {
    if (empty($term)) return e($text);
    $escaped_term = preg_quote(e($term), '/');
    return preg_replace('/(' . $escaped_term . ')/i', '<mark class="bg-yellow-200 dark:bg-yellow-400 dark:text-black rounded px-1">$1</mark>', e($text));
}

$page_title = 'All Development Projects | Genowa Delivery Unit';
require_once 'includes/header.php';
?>

<link href="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/css/tom-select.default.min.css" rel="stylesheet">
<style>
    .ts-control { border-radius: 0.5rem; border-color: #D1D5DB; padding: 0.65rem 1rem; font-size: 1rem; }
    .dark .ts-control { border-color: #4B5563; background-color: #374151; }
    .ts-dropdown .ts-option { padding: 0.75rem 1rem; font-size: 1rem; }
    .ts-control .ts-input, .dark .ts-control .ts-input { color: inherit; }
    .ts-dropdown { z-index: 20; } .dark .ts-dropdown { background-color: #1F2937; border-color: #4B5563; }
    .dark .ts-dropdown .active { background-color: #1E3A8A; }
    [data-status-pill]:focus { outline: 2px solid #FFB800; outline-offset: 2px; }
</style>

<div class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
    <div class="text-center mb-12">
        <h1 class="text-4xl md:text-5xl font-extrabold text-brand-blue dark:text-white">All Development Projects</h1>
        <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300">Use the filters below to explore projects by location, status, or manifesto pillar.</p>
    </div>

    <form id="filters-form" action="projects.php" method="get" class="bg-white dark:bg-dark-card p-6 lg:p-8 rounded-2xl shadow-xl mb-12 border border-gray-200 dark:border-gray-700">
        <div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-4 gap-6 items-end">
            <div>
                <label for="search" class="flex items-center text-sm font-semibold text-gray-800 dark:text-gray-200 mb-2">
                    <svg class="w-5 h-5 mr-2 text-gray-400" ...></svg>Keyword Search
                </label>
                <input type="text" name="search" id="search" value="<?php echo e($filters['search']); ?>" placeholder="e.g., 'Dispensary'" class="block w-full rounded-lg border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white shadow-sm focus:border-brand-purple focus:ring-brand-purple text-lg p-3">
            </div>
            <div>
                <label for="constituency" class="flex items-center text-sm font-semibold text-gray-800 dark:text-gray-200 mb-2">
                    <svg class="w-5 h-5 mr-2 text-gray-400" ...></svg>Constituency
                </label>
                <select name="constituency" id="constituency" placeholder="All Constituencies">
                    <option value="">All Constituencies</option>
                    <?php foreach ($constituencies as $c): ?>
                        <option value="<?php echo e($c['id']); ?>" <?php if ($filters['constituency'] == $c['id']) echo 'selected'; ?>><?php echo e($c['name']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="ward" class="flex items-center text-sm font-semibold text-gray-800 dark:text-gray-200 mb-2">
                    <svg class="w-5 h-5 mr-2 text-gray-400" ...></svg>Ward
                </label>
                <select name="ward" id="ward" placeholder="Select Constituency First" disabled></select>
            </div>
            <div>
                <label for="pillar" class="flex items-center text-sm font-semibold text-gray-800 dark:text-gray-200 mb-2">
                    <svg class="w-5 h-5 mr-2 text-gray-400" ...></svg>Manifesto Pillar
                </label>
                <select name="pillar" id="pillar" placeholder="All Pillars">
                    <option value="">All Pillars</option>
                    <?php foreach ($pillars as $p): ?>
                        <option value="<?php echo e($p['id']); ?>" <?php if ($filters['pillar'] == $p['id']) echo 'selected'; ?>><?php echo e($p['title']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        <div class="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
            <label class="flex items-center text-sm font-semibold text-gray-800 dark:text-gray-200 mb-3">
                <svg class="w-5 h-5 mr-2 text-gray-400" ...></svg>Project Status
            </label>
            <div class="flex flex-wrap gap-3">
                <?php $statuses = ['Planned', 'Ongoing', 'Completed', 'Stalled'];
                foreach ($statuses as $status): $checked = in_array($status, $filters['status']); ?>
                <label tabindex="0" data-status-pill class="relative inline-flex items-center cursor-pointer rounded-full px-4 py-2 text-sm font-medium transition-colors <?php echo $checked ? 'bg-brand-orange text-white shadow-md' : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-200 hover:bg-gray-200 dark:hover:bg-gray-600'; ?>">
                    <input type="checkbox" name="status[]" value="<?php echo e($status); ?>" <?php echo $checked ? 'checked' : ''; ?> class="sr-only">
                    <span><?php echo e($status); ?></span>
                </label>
                <?php endforeach; ?>
            </div>
        </div>
        <div class="mt-8 flex flex-col sm:flex-row items-center justify-end gap-4">
            <a href="projects.php" class="text-sm font-medium text-gray-600 dark:text-gray-300 hover:text-brand-orange transition-colors">Clear All Filters</a>
            <button type="submit" class="w-full sm:w-auto inline-flex justify-center items-center rounded-lg border border-transparent bg-brand-blue hover:bg-opacity-90 py-3 px-6 text-base font-medium text-white shadow-sm focus:outline-none focus:ring-2 focus:ring-brand-purple focus:ring-offset-2">
                <svg class="w-5 h-5 mr-2" ...></svg>Apply Filters
            </button>
        </div>
    </form>

    <?php
        $active_filters_exist = !empty($filters['search']) || !empty($filters['constituency']) || !empty($filters['ward']) || !empty($filters['status']) || !empty($filters['pillar']);
        if ($active_filters_exist):
    ?>
    <div class="mb-8 flex flex-wrap items-center gap-3">
        <h3 class="text-sm font-semibold text-gray-700 dark:text-gray-300 mr-2">Active Filters:</h3>
        <?php $current_params = $_GET; unset($current_params['page']); ?>

        <?php // Single value pills (search, constituency, ward, pillar)
        foreach(['search', 'constituency', 'ward', 'pillar'] as $key): if(!empty($filters[$key])):
            $temp_params = $current_params; unset($temp_params[$key]);
            $display_value = $filters[$key];
            if ($key === 'constituency') { foreach ($constituencies as $c) { if ($c['id'] == $filters[$key]) $display_value = $c['name']; } unset($temp_params['ward']); }
            if ($key === 'ward') { foreach ($all_wards as $w) { if ($w['id'] == $filters[$key]) $display_value = $w['name']; } }
            if ($key === 'pillar') { foreach ($pillars as $p) { if ($p['id'] == $filters[$key]) $display_value = $p['title']; } }
        ?>
            <a href="?<?php echo http_build_query($temp_params); ?>" title="Remove filter: <?php echo e($display_value); ?>" class="inline-flex items-center gap-x-2 rounded-full bg-gray-200 dark:bg-gray-700 px-3.5 py-1.5 text-sm font-medium text-gray-800 dark:text-gray-200 hover:bg-red-200 dark:hover:bg-red-800 transition-colors"><?php echo e($display_value); ?><svg class="h-4 w-4" ...></svg></a>
        <?php endif; endforeach; ?>

        <?php // Status pills (multi-value)
        if(!empty($filters['status'])): foreach($filters['status'] as $status_val):
            $temp_params = $current_params;
            if (isset($temp_params['status']) && is_array($temp_params['status'])) {
                $status_key = array_search($status_val, $temp_params['status']);
                if ($status_key !== false) { unset($temp_params['status'][$status_key]); }
            }
        ?>
            <a href="?<?php echo http_build_query($temp_params); ?>" title="Remove filter: <?php echo e($status_val); ?>" class="inline-flex items-center gap-x-2 rounded-full bg-blue-100 dark:bg-blue-900/50 px-3.5 py-1.5 text-sm font-medium text-blue-800 dark:text-blue-200 hover:bg-red-200 dark:hover:bg-red-800 transition-colors">Status: <?php echo e($status_val); ?><svg class="h-4 w-4" ...></svg></a>
        <?php endforeach; endif; ?>
    </div>
    <?php endif; ?>

    <div class="flex flex-col sm:flex-row justify-between items-baseline mb-6">
        <h2 class="text-2xl font-bold text-gray-800 dark:text-white mb-4 sm:mb-0">Results <span class="text-lg font-normal text-gray-500 dark:text-gray-400">(<?php echo number_format($total_projects); ?> Project<?php echo ($total_projects !== 1) ? 's' : ''; ?> Found)</span></h2>
    </div>

    <div class="mt-6 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
        <?php if (isset($db_error_message)): ?>
            <p class="col-span-full text-red-600"><?php echo e($db_error_message); ?></p>
        <?php elseif (empty($projects)): ?>
            <div class="col-span-full text-center bg-white dark:bg-dark-card p-12 rounded-lg shadow">
                <h3 class="text-lg font-medium text-gray-900 dark:text-white">No Projects Found</h3>
                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Try adjusting your filters or <a href="projects.php" class="text-brand-blue hover:underline">clearing them</a> to see all projects.</p>
            </div>
        <?php else: ?>
            <?php foreach ($projects as $project): ?>
            <a href="project.php?id=<?php echo e($project['id']); ?>" class="group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                <div class="relative">
                    <img loading="lazy" src="admin/uploads/<?php echo e($project['project_image'] ?? 'placeholder.jpg'); ?>" alt="Project image for <?php echo e($project['name']); ?>" class="h-56 w-full object-cover" onerror="this.onerror=null;this.src='https://placehold.co/600x400/1E3A8A/FFFFFF?text=Image+Not+Available';">
                    <div class="absolute top-3 right-3 px-2 py-1 text-xs font-semibold rounded-full text-white <?php switch ($project['status']) { case 'Completed': echo 'bg-green-600'; break; case 'Ongoing': echo 'bg-blue-600'; break; case 'Planned': echo 'bg-yellow-500'; break; case 'Stalled': echo 'bg-red-600'; break; }?>">
                        <?php echo e($project['status']); ?>
                    </div>
                </div>
                <div class="p-6">
                    <p class="text-sm font-semibold text-brand-orange uppercase"><?php echo highlight_term($project['ward_name'], $filters['search']); ?> Ward, <?php echo highlight_term($project['constituency_name'], $filters['search']); ?></p>
                    <h3 class="mt-2 text-xl font-bold text-brand-blue dark:text-white group-hover:text-brand-purple transition-colors duration-300"><?php echo highlight_term($project['name'], $filters['search']); ?></h3>
                    <p class="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-2 h-10"><?php echo highlight_term($project['description'], $filters['search']); ?></p>
                </div>
            </a>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    <div class="mt-12">
        <?php echo generate_pagination($page, $total_pages, $_GET); ?>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/js/tom-select.complete.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- TomSelect ---
    const allWards = <?php echo json_encode($all_wards); ?>;
    const selectedConstituency = '<?php echo e($filters['constituency']); ?>';
    const selectedWard = '<?php echo e($filters['ward']); ?>';

    const tomSelectSettings = { create: false };
    const constituencySelect = new TomSelect('#constituency', tomSelectSettings);
    const pillarSelect = new TomSelect('#pillar', tomSelectSettings);
    const wardSelect = new TomSelect('#ward', { ...tomSelectSettings, valueField: 'id', labelField: 'name', searchField: ['name'] });

    function populateWards(constituencyId, currentWardId) {
        wardSelect.clear();
        wardSelect.clearOptions();
        wardSelect.addOption({ id: '', name: 'All Wards' });
        if (!constituencyId) {
            wardSelect.disable();
            return;
        }
        wardSelect.enable();
        const relevantWards = allWards.filter(ward => String(ward.constituency_id) === String(constituencyId));
        relevantWards.forEach(ward => wardSelect.addOption({ id: ward.id, name: ward.name }));
        const isValidWard = relevantWards.some(ward => String(ward.id) === String(currentWardId));
        if (isValidWard) {
            wardSelect.setValue(currentWardId);
        }
    }
    populateWards(selectedConstituency, selectedWard);
    constituencySelect.on('change', function(value) { populateWards(value, ''); });

    // --- Status pill instant toggle + instant submit ---
    document.querySelectorAll('[data-status-pill]').forEach(function(pill) {
        pill.addEventListener('click', function(e) {
            if(e.target.tagName.toLowerCase() === 'input') return;
            const input = this.querySelector('input[type="checkbox"]');
            if (input) {
                input.checked = !input.checked;
                this.classList.toggle('bg-brand-orange', input.checked);
                this.classList.toggle('text-white', input.checked);
                this.classList.toggle('shadow-md', input.checked);
                this.classList.toggle('bg-gray-100', !input.checked);
                this.classList.toggle('dark:bg-gray-700', !input.checked);
                this.classList.toggle('text-gray-700', !input.checked);
                this.classList.toggle('dark:text-gray-200', !input.checked);
                this.classList.toggle('hover:bg-gray-200', !input.checked);
                this.classList.toggle('dark:hover:bg-gray-600', !input.checked);
                // Submit form instantly on pill toggle
                setTimeout(function(){
                    document.getElementById('filters-form').submit();
                }, 100);
            }
        });
        pill.addEventListener('keydown', function(e) {
            if (e.key === ' ' || e.key === 'Enter') {
                e.preventDefault();
                this.click();
            }
        });
    });

    // On load, fix pill class state
    document.querySelectorAll('[data-status-pill]').forEach(function(pill) {
        const input = pill.querySelector('input[type="checkbox"]');
        if (input && input.checked) {
            pill.classList.add('bg-brand-orange', 'text-white', 'shadow-md');
            pill.classList.remove('bg-gray-100', 'dark:bg-gray-700', 'text-gray-700', 'dark:text-gray-200', 'hover:bg-gray-200', 'dark:hover:bg-gray-600');
        }
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>
