<?php
// FILE: public_html/project.php
// PURPOSE: Displays all details for a single project with all enhancements.

require_once 'db_connect.php';
require_once 'admin/functions.php'; // For the e() helper function

// --- CONFIGURATION ---
define('RELATED_PROJECTS_LIMIT', 3);

// Start session for CSRF token on feedback form
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- GENERATE CSRF TOKEN ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// --- Get Project ID and Fetch Data ---
$project_id = $_GET['id'] ?? null;
if (!$project_id || !filter_var($project_id, FILTER_VALIDATE_INT)) {
    header("Location: projects.php");
    exit;
}

$project = null;
$photos = [];
$related_projects = [];
$approved_feedback = [];

try {
    // Fetch project details along with location and pillar info
    $sql = "SELECT 
                p.*,
                w.name AS ward_name,
                c.name AS constituency_name,
                mp.title AS pillar_title,
                mp.description as pillar_description
            FROM projects p
            JOIN wards w ON p.ward_id = w.id
            JOIN constituencies c ON w.constituency_id = c.id
            JOIN manifesto_pillars mp ON p.manifesto_pillar_id = mp.id
            WHERE p.id = :id AND p.is_approved = 1"; // Ensure project is approved
            
    $stmt = $pdo->prepare($sql);
    $stmt->execute(['id' => $project_id]);
    $project = $stmt->fetch();

    // If project not found, redirect
    if (!$project) {
        header("Location: projects.php");
        exit;
    }

    // --- Calculate Progress for Ongoing Projects ---
    $completion_percentage = 0;
    if ($project['status'] === 'Completed') {
        $completion_percentage = 100;
    } elseif ($project['status'] === 'Ongoing' && $project['start_date'] && $project['expected_completion_date']) {
        $today = new DateTime();
        $start_date = new DateTime($project['start_date']);
        $expected_date = new DateTime($project['expected_completion_date']);
        
        if ($start_date <= $expected_date && $today >= $start_date) {
            $total_duration = $start_date->diff($expected_date)->days;
            $elapsed_duration = $start_date->diff($today)->days;
            $completion_percentage = ($total_duration > 0) ? min(100, floor(($elapsed_duration / $total_duration) * 100)) : 100;
        }
    }

    // --- Fetch Related Projects (from the same ward) ---
    $related_sql = "SELECT p.id, p.name, w.name as ward_name, 
                    (SELECT image_url FROM project_photos pp WHERE pp.project_id = p.id ORDER BY pp.id DESC LIMIT 1) as project_image
                    FROM projects p
                    JOIN wards w ON p.ward_id = w.id
                    WHERE p.ward_id = :ward_id AND p.id != :current_project_id AND p.is_approved = 1
                    LIMIT " . RELATED_PROJECTS_LIMIT;
    $related_stmt = $pdo->prepare($related_sql);
    $related_stmt->execute(['ward_id' => $project['ward_id'], 'current_project_id' => $project_id]);
    $related_projects = $related_stmt->fetchAll();

    // Fetch all photos for this project
    $photos_stmt = $pdo->prepare("SELECT * FROM project_photos WHERE project_id = :project_id ORDER BY photo_type, uploaded_at DESC");
    $photos_stmt->execute(['project_id' => $project_id]);
    $photos = $photos_stmt->fetchAll();
    
    // --- Fetch Approved Feedback ---
    $feedback_stmt = $pdo->prepare("SELECT * FROM project_feedback WHERE project_id = :project_id AND is_approved = 1 ORDER BY submitted_at DESC");
    $feedback_stmt->execute(['project_id' => $project_id]);
    $approved_feedback = $feedback_stmt->fetchAll();


} catch (PDOException $e) {
    $db_error_message = "Could not load project details due to a technical issue.";
    error_log("Project Detail Page DB Error: " . $e->getMessage());
}

$page_title = $project
    ? $project['name'] . " in " . $project['ward_name'] . " Ward, " . $project['constituency_name'] . " | Genowa Delivery Unit"
    : "Project Details | Genowa Delivery Unit";

$meta_description = $project
    ? mb_substr(strip_tags($project['description']), 0, 160)
    : "Detailed information on a development project in Homa Bay County as tracked by the Genowa Delivery Unit.";

$canonical_url = "https://www.genowa.co.ke/project.php?id=" . urlencode($project_id);

$seo_img_url = null;
if (!empty($photos)) {
    foreach (['After', 'Progress', 'Before'] as $ptype) {
        foreach ($photos as $photo) {
            if ($photo['photo_type'] === $ptype && !empty($photo['image_url'])) {
                $seo_img_url = "https://www.genowa.co.ke/admin/uploads/" . $photo['image_url'];
                break 2;
            }
        }
    }
}
$og_image = $seo_img_url ?? "https://www.genowa.co.ke/admin/assets/genowa.png";
$twitter_image = $og_image;
$og_description = $meta_description;
$og_url = $canonical_url;
$twitter_description = $meta_description;
include 'includes/header.php';
?>

<!-- NEW: Styles for the lightbox -->
<style>
    .lightbox {
        display: none;
        position: fixed;
        z-index: 9999;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.85);
        justify-content: center;
        align-items: center;
        padding: 20px;
        box-sizing: border-box;
    }
    .lightbox img {
        max-width: 95%;
        max-height: 90%;
        border-radius: 8px;
        box-shadow: 0 0 30px rgba(0,0,0,0.5);
    }
    .lightbox-close {
        position: absolute;
        top: 20px;
        right: 30px;
        font-size: 40px;
        color: white;
        cursor: pointer;
        font-weight: bold;
    }
</style>

<?php if (isset($db_error_message)): ?>
    <div class="text-center bg-red-100 p-4 my-8 container mx-auto rounded-lg">
        <p class="text-red-700"><?php echo e($db_error_message); ?></p>
    </div>
<?php elseif ($project): ?>
    <main class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <!-- Project Header -->
        <div class="mb-8 border-b pb-8">
            <p class="text-lg font-semibold text-brand-orange uppercase"><?php echo e($project['constituency_name']); ?> Constituency &gt; <?php echo e($project['ward_name']); ?> Ward</p>
            <h1 class="mt-2 text-4xl md:text-5xl font-extrabold text-brand-blue"><?php echo e($project['name']); ?></h1>
            <div class="mt-4 px-3 py-1 text-sm font-bold inline-block rounded-full text-white
                <?php 
                    switch ($project['status']) {
                        case 'Completed': echo 'bg-green-600'; break;
                        case 'Ongoing': echo 'bg-blue-600'; break;
                        case 'Planned': echo 'bg-yellow-500 text-gray-800'; break;
                        case 'Stalled': echo 'bg-red-600'; break;
                    }
                ?>">
                Status: <?php echo e($project['status']); ?>
            </div>
            
            <?php if ($project['status'] !== 'Planned' && $completion_percentage > 0): ?>
                <div class="mt-6">
                    <p class="text-sm font-medium text-gray-600 mb-1">Timeline Progress</p>
                    <div class="w-full bg-gray-200 rounded-full h-4">
                        <div class="bg-brand-orange h-4 rounded-full text-xs font-medium text-white text-center p-0.5 leading-none" style="width: <?php echo $completion_percentage; ?>%">
                        </div>
                    </div>
                    <div class="text-right text-sm font-semibold mt-1 text-brand-orange">
                        <?php echo $completion_percentage; ?>% Complete
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <!-- Main Content Grid -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-12">
            <!-- Left Column: Details -->
            <div class="lg:col-span-2 space-y-12">
                <!-- Photo Gallery Section -->
                <section>
                    <h2 class="text-2xl font-bold text-gray-800 mb-4">Project Gallery</h2>
                    <?php 
                        $before_img = null;
                        $after_img = null;
                        $progress_photos = [];
                        foreach ($photos as $photo) {
                            if ($photo['photo_type'] == 'Before' && !$before_img) $before_img = $photo;
                            if ($photo['photo_type'] == 'After' && !$after_img) $after_img = $photo;
                            if ($photo['photo_type'] == 'Progress') $progress_photos[] = $photo;
                        }
                    ?>
                    <?php if ($before_img && $after_img): ?>
                        <div x-data="{ value: 50 }" class="w-full relative rounded-lg overflow-hidden shadow-lg mb-8">
                            <!-- After image -->
                            <div class="absolute inset-0 cursor-pointer" onclick="openLightbox('admin/uploads/<?php echo e($after_img['image_url']); ?>')">
                                <img src="admin/uploads/<?php echo e($after_img['image_url']); ?>" alt="After: <?php echo e($after_img['caption']); ?>" class="w-full h-full object-cover" loading="lazy">
                                <div class="absolute top-2 right-2 bg-green-600 text-white text-xs font-bold px-2 py-1 rounded-full">AFTER</div>
                            </div>
                            <!-- Before image -->
                            <div class="absolute inset-0" :style="`clip-path: polygon(0 0, ${value}% 0, ${value}% 100%, 0 100%)`">
                                <div class="absolute inset-0 cursor-pointer" onclick="openLightbox('admin/uploads/<?php echo e($before_img['image_url']); ?>')">
                                    <img src="admin/uploads/<?php echo e($before_img['image_url']); ?>" alt="Before: <?php echo e($before_img['caption']); ?>" class="w-full h-full object-cover" loading="lazy">
                                    <div class="absolute top-2 left-2 bg-red-600 text-white text-xs font-bold px-2 py-1 rounded-full">BEFORE</div>
                                </div>
                            </div>
                            <!-- Slider -->
                            <input type="range" min="0" max="100" x-model="value" class="absolute inset-0 w-full h-full cursor-pointer opacity-0">
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($progress_photos) || (!$before_img || !$after_img)): ?>
                        <div class="grid grid-cols-2 sm:grid-cols-3 gap-4">
                            <?php if (!$before_img || !$after_img): // Show Before/After separately if slider isn't used ?>
                                <?php if ($before_img): ?> <div><img src="admin/uploads/<?php echo e($before_img['image_url']); ?>" class="rounded-lg shadow-md object-cover h-48 w-full cursor-pointer" onclick="openLightbox('admin/uploads/<?php echo e($before_img['image_url']); ?>')" loading="lazy"><p class="text-center text-sm mt-1">Before</p></div> <?php endif; ?>
                                <?php if ($after_img): ?> <div><img src="admin/uploads/<?php echo e($after_img['image_url']); ?>" class="rounded-lg shadow-md object-cover h-48 w-full cursor-pointer" onclick="openLightbox('admin/uploads/<?php echo e($after_img['image_url']); ?>')" loading="lazy"><p class="text-center text-sm mt-1">After</p></div> <?php endif; ?>
                            <?php endif; ?>
                            <?php foreach ($progress_photos as $photo): ?>
                                <div><img src="admin/uploads/<?php echo e($photo['image_url']); ?>" class="rounded-lg shadow-md object-cover h-48 w-full cursor-pointer" onclick="openLightbox('admin/uploads/<?php echo e($photo['image_url']); ?>')" loading="lazy"><p class="text-center text-sm mt-1"><?php echo e($photo['caption'] ?: 'Progress'); ?></p></div>
                            <?php endforeach; ?>
                        </div>
                    <?php elseif (!$before_img && !$after_img && empty($progress_photos)): ?>
                         <div class="text-center bg-gray-100 p-8 rounded-lg">
                             <p class="text-gray-500">No photos have been uploaded for this project yet.</p>
                         </div>
                    <?php endif; ?>
                </section>

                <!-- Description Section -->
                <section>
                     <h2 class="text-2xl font-bold text-gray-800 mb-4">Project Description</h2>
                     <div class="prose max-w-none text-gray-700">
                         <?php echo nl2br(e($project['description'])); ?>
                     </div>
                </section>

                 <!-- Beneficiaries Section -->
                <section>
                     <h2 class="text-2xl font-bold text-gray-800 mb-4">Beneficiaries</h2>
                     <div class="prose max-w-none text-gray-700">
                         <?php echo nl2br(e($project['beneficiaries_summary'])); ?>
                     </div>
                </section>
            </div>

            <!-- Right Column: Key Facts -->
            <aside class="lg:col-span-1 space-y-8">
                <div class="bg-white p-6 rounded-lg shadow-lg border-l-4 border-brand-orange">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Key Facts</h3>
                    <ul class="space-y-4">
                        <li class="flex justify-between items-center border-b pb-2">
                            <span class="font-semibold text-gray-600">Contractor:</span>
                            <span class="text-gray-800 text-right"><?php echo e($project['contractor_name'] ?: 'N/A'); ?></span>
                        </li>
                         <li class="flex justify-between items-center border-b pb-2">
                            <span class="font-semibold text-gray-600">Start Date:</span>
                            <span class="text-gray-800"><?php echo $project['start_date'] ? date('M j, Y', strtotime($project['start_date'])) : 'N/A'; ?></span>
                        </li>
                        <li class="flex justify-between items-center border-b pb-2">
                            <span class="font-semibold text-gray-600">Expected Completion:</span>
                            <span class="text-gray-800"><?php echo $project['expected_completion_date'] ? date('M j, Y', strtotime($project['expected_completion_date'])) : 'N/A'; ?></span>
                        </li>
                         <li class="flex justify-between items-center border-b pb-2">
                            <span class="font-semibold text-gray-600">Actual Completion:</span>
                            <span class="text-gray-800"><?php echo $project['actual_completion_date'] ? date('M j, Y', strtotime($project['actual_completion_date'])) : 'N/A'; ?></span>
                        </li>
                    </ul>
                </div>
                <div class="bg-white p-6 rounded-lg shadow-lg border-l-4 border-brand-purple">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Financials</h3>
                    <ul class="space-y-4">
                         <li class="flex justify-between items-center border-b pb-2">
                            <span class="font-semibold text-gray-600">Budget Allocated:</span>
                            <span class="font-bold text-gray-800">KES <?php echo number_format($project['budget_allocated'], 2); ?></span>
                        </li>
                        <li class="flex justify-between items-center">
                            <span class="font-semibold text-gray-600">Budget Spent:</span>
                            <span class="font-bold text-gray-800">KES <?php echo number_format($project['budget_spent'] ?? 0, 2); ?></span>
                        </li>
                    </ul>
                </div>
                 <div class="bg-white p-6 rounded-lg shadow-lg border-l-4 border-brand-blue">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Manifesto Linkage</h3>
                    <p class="font-bold text-brand-blue"><?php echo e($project['pillar_title']); ?></p>
                    <p class="mt-2 text-gray-600"><?php echo e($project['pillar_description']); ?></p>
                </div>
                
                <div class="bg-white p-6 rounded-lg shadow-lg">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Share This Project</h3>
                    <div class="flex space-x-2">
                        <?php
                            $share_url = urlencode("https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
                            $share_text = urlencode("Check out this project in Homa Bay: " . $project['name']);
                        ?>
                        <a href="https://wa.me/?text=<?php echo $share_text . '%20' . $share_url; ?>" target="_blank" class="flex-1 text-center bg-green-500 text-white font-bold py-2 px-4 rounded-lg hover:bg-green-600 transition-colors">WhatsApp</a>
                        <a href="https://twitter.com/intent/tweet?url=<?php echo $share_url; ?>&text=<?php echo $share_text; ?>" target="_blank" class="flex-1 text-center bg-blue-400 text-white font-bold py-2 px-4 rounded-lg hover:bg-blue-500 transition-colors">Twitter</a>
                        <a href="https://www.facebook.com/sharer/sharer.php?u=<?php echo $share_url; ?>" target="_blank" class="flex-1 text-center bg-blue-800 text-white font-bold py-2 px-4 rounded-lg hover:bg-blue-900 transition-colors">Facebook</a>
                    </div>
                </div>

            </aside>
        </div>

        <!-- Public Feedback Section -->
        <section id="feedback-section" class="mt-16 border-t pt-12">
            <h2 class="text-3xl font-extrabold text-brand-blue mb-8">Public Feedback & Comments</h2>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-12">
                <!-- Leave a comment form -->
                <div class="bg-white p-6 rounded-lg shadow-lg">
                    <h3 class="text-xl font-bold mb-4">Leave Your Feedback</h3>
                    <!-- NEW: AJAX form submission -->
                    <form id="feedback-form" class="space-y-4">
                        <input type="hidden" name="project_id" value="<?php echo e($project['id']); ?>">
                        <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                        <div>
                            <label for="author_name" class="block text-sm font-medium text-gray-700">Your Name</label>
                            <input type="text" name="author_name" id="author_name" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3 text-base">
                        </div>
                        <div>
                            <label for="comment" class="block text-sm font-medium text-gray-700">Your Comment</label>
                            <textarea name="comment" id="comment" rows="4" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3 text-base"></textarea>
                        </div>
                        <div>
                            <button type="submit" id="submit-feedback-btn" class="w-full inline-flex justify-center rounded-md border border-transparent bg-brand-orange py-3 px-6 text-base font-medium text-white shadow-sm hover:bg-opacity-90 disabled:bg-opacity-50">Submit Comment</button>
                        </div>
                    </form>
                    <div id="feedback-message" class="mt-4"></div>
                </div>
                <!-- Display approved comments -->
                <div class="space-y-6">
                    <?php if (empty($approved_feedback)): ?>
                        <div class="text-center text-gray-500 p-6 bg-gray-50 rounded-lg">
                            <p>No comments have been posted for this project yet. Be the first to leave your feedback!</p>
                        </div>
                    <?php else: ?>
                        <?php foreach($approved_feedback as $feedback): ?>
                            <div class="bg-white p-4 rounded-lg shadow">
                                <div class="flex items-center justify-between">
                                    <p class="text-sm font-bold text-brand-blue"><?php echo e($feedback['author_name']); ?></p>
                                    <p class="text-xs text-gray-500"><?php echo date('M j, Y', strtotime($feedback['submitted_at'])); ?></p>
                                </div>
                                <p class="mt-2 text-gray-700"><?php echo nl2br(e($feedback['comment'])); ?></p>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </section>


        <!-- Related Projects Section -->
        <?php if (!empty($related_projects)): ?>
        <section class="mt-20 border-t pt-12">
             <h2 class="text-3xl font-extrabold text-brand-blue mb-8">More Projects in <?php echo e($project['ward_name']); ?> Ward</h2>
             <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($related_projects as $related_project): ?>
                    <a href="project.php?id=<?php echo e($related_project['id']); ?>" class="group block bg-white rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                        <div class="relative">
                            <img 
                                src="admin/uploads/<?php echo e($related_project['project_image'] ?? 'placeholder.jpg'); ?>" 
                                alt="Project image for <?php echo e($related_project['name']); ?>" 
                                class="h-56 w-full object-cover"
                                loading="lazy"
                                onerror="this.onerror=null;this.src='https://placehold.co/600x400/1E3A8A/FFFFFF?text=Image+Not+Available';"
                            >
                        </div>
                        <div class="p-6">
                            <p class="text-sm font-semibold text-brand-orange uppercase"><?php echo e($related_project['ward_name']); ?> Ward</p>
                            <h3 class="mt-2 text-xl font-bold text-brand-blue group-hover:text-brand-purple transition-colors duration-300">
                                <?php echo e($related_project['name']); ?>
                            </h3>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        </section>
        <?php endif; ?>
    </main>
<?php else: ?>
    <div class="text-center py-20 container mx-auto">
        <h1 class="text-3xl font-bold text-red-600">Project Not Found</h1>
        <p class="mt-4 text-gray-600">The project you are looking for does not exist or may have been moved.</p>
        <a href="projects.php" class="mt-6 inline-block bg-brand-orange text-white font-bold py-3 px-6 rounded-full hover:bg-opacity-90">Back to All Projects</a>
    </div>
<?php endif; ?>

<!-- NEW: Lightbox element -->
<div id="lightbox" class="lightbox" onclick="closeLightbox()">
    <span class="lightbox-close">&times;</span>
    <img id="lightbox-img" src="">
</div>

<!-- AlpineJS for the slider -->
<script src="//unpkg.com/alpinejs" defer></script>

<!-- NEW: JavaScript for lightbox and AJAX form -->
<script>
    // --- Lightbox Logic ---
    const lightbox = document.getElementById('lightbox');
    const lightboxImg = document.getElementById('lightbox-img');

    function openLightbox(src) {
        lightboxImg.src = src;
        lightbox.style.display = 'flex';
    }

    function closeLightbox() {
        lightbox.style.display = 'none';
        lightboxImg.src = '';
    }

    // --- AJAX Feedback Form Logic ---
    const feedbackForm = document.getElementById('feedback-form');
    const feedbackMessageDiv = document.getElementById('feedback-message');
    const submitBtn = document.getElementById('submit-feedback-btn');

    feedbackForm.addEventListener('submit', function(e) {
        e.preventDefault();
        submitBtn.disabled = true;
        submitBtn.textContent = 'Submitting...';
        feedbackMessageDiv.innerHTML = '';

        const formData = new FormData(feedbackForm);

        fetch('submit_feedback.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                feedbackMessageDiv.innerHTML = `<div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md"><p class="font-bold">Success!</p><p>${data.message}</p></div>`;
                feedbackForm.reset();
            } else {
                feedbackMessageDiv.innerHTML = `<div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md"><p class="font-bold">Error</p><p>${data.message}</p></div>`;
            }
        })
        .catch(error => {
            feedbackMessageDiv.innerHTML = `<div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md"><p class="font-bold">Error</p><p>A network error occurred. Please try again.</p></div>`;
            console.error('Error:', error);
        })
        .finally(() => {
            submitBtn.disabled = false;
            submitBtn.textContent = 'Submit Comment';
        });
    });
</script>

<?php include 'includes/footer.php'; ?>
