<?php
// FILE: public_html/join.php
// PURPOSE: "Join the Genowa Movement" page with sign-up form and processing logic.

require_once 'db_connect.php';
require_once 'admin/functions.php';

$errors = [];
$success_message = '';
$inputs = [];
$new_membership_id = null;

// Fetch wards for the dropdown
try {
    $wards = $pdo->query("SELECT id, name FROM wards ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db'] = "Could not fetch ward list. Please contact an administrator.";
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitize and store inputs to re-populate the form on error
    $inputs['name'] = trim($_POST['name']);
    $inputs['email'] = trim($_POST['email']);
    $inputs['phone_number'] = trim($_POST['phone_number']);
    $inputs['ward_id'] = trim($_POST['ward_id']);
    
    // --- Validation ---
    if (empty($inputs['name'])) $errors['name'] = "Your full name is required.";
    if (empty($inputs['ward_id'])) $errors['ward_id'] = "Please select your home ward.";
    
    if (empty($inputs['email'])) {
        $errors['email'] = "Your email address is required.";
    } elseif (!filter_var($inputs['email'], FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = "Please enter a valid email address.";
    }
    
    if (empty($inputs['phone_number'])) {
        $errors['phone_number'] = "Your phone number is required.";
    } elseif (!preg_match('/^(?:\+254|0)?(7|1)\d{8}$/', $inputs['phone_number'])) {
        $errors['phone_number'] = "Please enter a valid Kenyan phone number (e.g., 0712345678).";
    }

    // --- Check for duplicates if no validation errors so far ---
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM movement_members WHERE email = :email OR phone_number = :phone_number");
            $stmt->execute(['email' => $inputs['email'], 'phone_number' => $inputs['phone_number']]);
            if ($stmt->fetch()) {
                $errors['duplicate'] = "An account with this email or phone number already exists in the movement.";
            }
        } catch (PDOException $e) {
            $errors['db'] = "A database error occurred during validation.";
        }
    }

    // --- Process registration if no errors ---
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Get the last membership_id and increment it
            $last_id_stmt = $pdo->query("SELECT MAX(membership_id) as max_id FROM movement_members");
            $last_id = $last_id_stmt->fetchColumn();
            $new_membership_id = $last_id ? $last_id + 1 : 5001;

            // Prepare SQL to insert new member
            $sql = "INSERT INTO movement_members (membership_id, name, email, phone_number, ward_id) VALUES (:membership_id, :name, :email, :phone_number, :ward_id)";
            $stmt = $pdo->prepare($sql);
            
            $stmt->execute([
                'membership_id' => $new_membership_id,
                'name' => $inputs['name'],
                'email' => $inputs['email'],
                'phone_number' => $inputs['phone_number'],
                'ward_id' => $inputs['ward_id']
            ]);

            $pdo->commit();

            // --- START: Send Notifications ---
            // 1. Send notification to the new member
            $sms_message = "Karibu to the Genowa Movement! Your membership ID is " . $new_membership_id . ". Together, we'll unlock Homa Bay's endless potential. #GladysWangaDelivers";
            send_sms($inputs['phone_number'], $sms_message);
            
            $email_subject = "Welcome to the Genowa Movement!";
            $email_body = "<p>Dear " . e($inputs['name']) . ",</p><p>Welcome to the Genowa Movement! Your official Membership ID is: <strong>" . $new_membership_id . "</strong></p><p>Thank you for believing in the endless potential of Homa Bay County!</p>";
            send_email($inputs['email'], $email_subject, $email_body);
            
            // 2. Send notification to admin
            $admin_email = "projects@genowa.co.ke";
            $admin_subject = "New Member Alert: Genowa Movement";
            $admin_body = "<p>A new member has joined the Genowa Movement.</p><p><strong>Name:</strong> ".e($inputs['name'])."</p><p><strong>Membership ID:</strong> ".$new_membership_id."</p>";
            send_email($admin_email, $admin_subject, $admin_body);
            // --- END: Send Notifications ---

            $success_message = "Welcome to the movement! Your official Membership ID is:";
            $inputs = []; // Clear form on success

        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors['db'] = "An error occurred during registration. Please try again. Error: " . $e->getMessage();
        }
    }
}

include 'includes/header.php';
?>

<div class="bg-brand-blue">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-16 text-center">
        <h1 class="text-4xl md:text-5xl font-extrabold text-white">Join the Genowa Movement</h1>
        <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-300">
            Be part of the transformation. Join a community of citizens committed to tracking progress and ensuring accountability in Homa Bay County.
        </p>
    </div>
</div>

<main class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
    <div class="max-w-xl mx-auto">
        <div class="bg-white rounded-lg shadow-xl p-8">
            <?php if ($success_message): ?>
                <div class="text-center p-6 bg-green-50 border-l-4 border-green-500">
                    <h3 class="text-lg font-bold text-green-800"><?php echo e($success_message); ?></h3>
                    <p class="mt-2 text-5xl font-black text-brand-purple tracking-tight"><?php echo e($new_membership_id); ?></p>
                    <p class="mt-4 text-gray-600">A confirmation has been sent to your email and phone. Thank you for believing in Homa Bay!</p>
                </div>
            <?php else: ?>
                <form action="join.php" method="post" class="space-y-6">
                    <?php if (!empty($errors)): ?>
                        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                            <p class="font-bold">Please correct the errors below:</p>
                            <ul class="list-disc list-inside mt-2">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo e($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <div>
                        <label for="name" class="block text-sm font-medium text-gray-700">Full Name</label>
                        <div class="relative mt-1 rounded-md shadow-sm">
                            <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z" /></svg></div>
                            <input type="text" name="name" id="name" value="<?php echo e($inputs['name'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3">
                        </div>
                    </div>
                     <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                           <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                           <div class="relative mt-1 rounded-md shadow-sm">
                               <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M21.75 6.75v10.5a2.25 2.25 0 01-2.25 2.25h-15a2.25 2.25 0 01-2.25-2.25V6.75m19.5 0A2.25 2.25 0 0019.5 4.5h-15a2.25 2.25 0 00-2.25 2.25m19.5 0v.243a2.25 2.25 0 01-1.07 1.916l-7.5 4.615a2.25 2.25 0 01-2.36 0L3.32 8.91a2.25 2.25 0 01-1.07-1.916V6.75" /></svg></div>
                               <input type="email" name="email" id="email" value="<?php echo e($inputs['email'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3">
                           </div>
                        </div>
                         <div>
                           <label for="phone_number" class="block text-sm font-medium text-gray-700">Phone Number</label>
                            <div class="relative mt-1 rounded-md shadow-sm">
                               <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M2.25 6.75c0 8.284 6.716 15 15 15h2.25a2.25 2.25 0 002.25-2.25v-1.372c0-.516-.351-.966-.852-1.091l-4.423-1.106c-.44-.11-.902.055-1.173.417l-.97 1.293c-.282.376-.769.542-1.21.38a12.035 12.035 0 01-7.143-7.143c-.162-.441.004-.928.38-1.21l1.293-.97c.363-.271.527-.734.417-1.173L6.963 3.102a1.125 1.125 0 00-1.091-.852H4.5A2.25 2.25 0 002.25 4.5v2.25z" /></svg></div>
                               <input type="tel" name="phone_number" id="phone_number" value="<?php echo e($inputs['phone_number'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3" placeholder="0712345678">
                            </div>
                        </div>
                    </div>
                    <div>
                        <label for="ward_id" class="block text-sm font-medium text-gray-700">Your Ward</label>
                        <select name="ward_id" id="ward_id" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3 text-base">
                            <option value="">-- Please Select --</option>
                            <?php if(!empty($wards)): ?>
                                <?php foreach($wards as $ward): ?>
                                <option value="<?php echo e($ward['id']); ?>" <?php if(($inputs['ward_id'] ?? '') == $ward['id']) echo 'selected'; ?>><?php echo e($ward['name']); ?></option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                    </div>
                    <div class="pt-2">
                        <button type="submit" class="w-full inline-flex justify-center rounded-md border border-transparent bg-brand-orange py-3 px-6 text-base font-medium text-white shadow-sm hover:bg-opacity-90 transition-transform duration-200 hover:scale-105">
                            Join the Movement
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php include 'includes/footer.php'; ?>
