<?php
// FILE: public_html/data_dashboard.php
// --- CORRECTED & UNABRIDGED ---
// --- UPDATED (2025-07-24) ---: Enhanced PDF export with better page-break logic for proper space utilization.

require_once 'db_connect.php';
require_once 'admin/functions.php';

$constituencies = $pdo->query("SELECT id, name FROM constituencies ORDER BY name ASC")->fetchAll();
$all_wards = $pdo->query("SELECT id, name, constituency_id FROM wards ORDER BY name ASC")->fetchAll();
$pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY id ASC")->fetchAll();

include 'includes/header.php';
?>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
<style>
    .loader { border: 4px solid #f3f3f3; border-radius: 50%; border-top: 4px solid #F97316; width: 40px; height: 40px; animation: spin 1s linear infinite; }
    @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    .chart-container { page-break-inside: avoid; } /* For better PDF printing */
</style>

<div class="bg-brand-blue">
  <div class="container mx-auto px-4 py-16 text-center">
    <h1 class="text-4xl font-extrabold text-white">County Data Dashboard</h1>
    <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-300">
      An interactive overview of development projects across Homa Bay County.
    </p>
  </div>
</div>

<main class="container mx-auto px-4 py-12">
  <div class="bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg mb-6">
    <form id="filter-form">
      <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 items-end">
        <div>
          <label for="constituency_id" class="block text-sm font-medium text-gray-700 dark:text-gray-200">Constituency</label>
          <select name="constituency_id" id="constituency_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm">
            <option value="">All Constituencies</option>
            <?php foreach ($constituencies as $c): ?>
              <option value="<?= $c['id'] ?>"><?= e($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label for="ward_id" class="block text-sm font-medium text-gray-700 dark:text-gray-200">Ward</label>
          <select name="ward_id" id="ward_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm" disabled>
            <option value="">All Wards</option>
          </select>
        </div>
        <div>
          <label for="pillar_id" class="block text-sm font-medium text-gray-700 dark:text-gray-200">Sector (Pillar)</label>
          <select name="pillar_id" id="pillar_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm">
            <option value="">All Sectors</option>
            <?php foreach ($pillars as $p): ?>
              <option value="<?= $p['id'] ?>"><?= e($p['title']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="flex items-center space-x-2">
          <button type="submit" class="w-full bg-brand-orange text-white px-4 py-2 rounded-md font-semibold hover:bg-opacity-90">Apply Filter</button>
          <button type="button" id="reset-filters" class="w-full bg-gray-200 px-4 py-2 rounded-md font-semibold text-gray-700 hover:bg-gray-300">Reset</button>
        </div>
      </div>
    </form>
  </div>

  <div class="text-right mb-4">
    <button id="open-pdf-preview" class="bg-brand-purple text-white px-4 py-2 rounded-md font-semibold hover:bg-opacity-90">Download Report</button>
  </div>

  <div id="dashboard-container" class="relative">
      <div id="loading-overlay" class="absolute inset-0 bg-white/70 dark:bg-dark-card/70 z-10 flex items-center justify-center hidden">
          <div class="loader"></div>
      </div>
      <div id="dashboard-content">
          <div id="dashboard-summary" class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8"></div>
          <div id="charts" class="grid grid-cols-1 lg:grid-cols-2 gap-8"></div>
      </div>
  </div>
</main>

<div id="pdf-preview-modal" class="fixed inset-0 bg-gray-800 bg-opacity-75 z-50 hidden flex items-center justify-center">
    <div class="bg-white rounded-lg shadow-2xl w-full max-w-4xl h-[90vh] flex flex-col">
        <div class="p-4 border-b flex justify-between items-center">
            <h3 class="text-xl font-bold">PDF Report Preview</h3>
            <button id="close-pdf-preview" class="text-2xl">&times;</button>
        </div>
        <div id="pdf-preview-body" class="p-8 overflow-y-auto flex-grow bg-gray-50">
            </div>
        <div class="p-4 border-t bg-gray-100 flex justify-end">
            <button id="confirm-pdf-download" class="bg-green-600 text-white px-6 py-2 rounded-md font-semibold">Confirm & Download</button>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const allWards = <?= json_encode($all_wards); ?>;
    const form = document.getElementById('filter-form');
    const constituencySelect = document.getElementById('constituency_id');
    const wardSelect = document.getElementById('ward_id');
    const summaryDiv = document.getElementById('dashboard-summary');
    const chartsDiv = document.getElementById('charts');
    const loadingOverlay = document.getElementById('loading-overlay');
    let charts = {}; // To hold chart instances for smooth updates
    let currentReportTitle = 'The Entire County';

    // --- Dependent Dropdown Logic ---
    constituencySelect.addEventListener('change', function() {
        const constituencyId = this.value;
        wardSelect.innerHTML = '<option value="">All Wards</option>';
        wardSelect.disabled = true;
        if (constituencyId) {
            const relevantWards = allWards.filter(ward => ward.constituency_id == constituencyId);
            relevantWards.forEach(ward => {
                const opt = document.createElement('option');
                opt.value = ward.id;
                opt.textContent = ward.name;
                wardSelect.appendChild(opt);
            });
            wardSelect.disabled = false;
        }
    });

    // --- Chart Rendering and Updating ---
    function destroyCharts() {
        for (const chartId in charts) {
            if (charts[chartId]) {
                charts[chartId].destroy();
            }
        }
        charts = {};
        chartsDiv.innerHTML = ''; // Clear canvas elements
    }

    function updateDashboard(data) {
        destroyCharts();
        currentReportTitle = data.report_title;

        // Render KPI Summary
        summaryDiv.innerHTML = `
            <div class='bg-white dark:bg-dark-card p-5 rounded-lg shadow-lg text-center'><p class='text-sm font-bold text-gray-500 uppercase'>Total Projects</p><p class='mt-1 text-4xl font-extrabold text-brand-purple'>${Number(data.kpis.total_projects).toLocaleString()}</p></div>
            <div class='bg-white dark:bg-dark-card p-5 rounded-lg shadow-lg text-center'><p class='text-sm font-bold text-gray-500 uppercase'>Total Investment</p><p class='mt-1 text-4xl font-extrabold text-brand-purple'>KES ${((data.kpis.total_investment || 0) / 1000000).toFixed(1)}M</p></div>
            <div class='bg-white dark:bg-dark-card p-5 rounded-lg shadow-lg text-center'><p class='text-sm font-bold text-gray-500 uppercase'>Avg. Delay (Completed)</p><p class='mt-1 text-4xl font-extrabold text-brand-purple'>${data.kpis.avg_delay} Days</p></div>
        `;

        // Add HTML containers for charts
        chartsDiv.innerHTML = `
            <div class="bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Projects by Status</h3><div class="h-64"><canvas id="statusChart"></canvas></div></div>
            <div class="bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Projects by Sector</h3><div class="h-64"><canvas id="sectorChart"></canvas></div></div>
            <div class="bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Project Timeliness (Completed)</h3><div class="h-64"><canvas id="timelinessChart"></canvas></div></div>
            <div class="bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Issue Resolution Status</h3><div class="h-64"><canvas id="issueStatusChart"></canvas></div></div>
            <div class="lg:col-span-2 bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Budget Utilization by Sector (KES)</h3><div class="h-80"><canvas id="budgetUtilizationChart"></canvas></div></div>
            <div class="lg:col-span-2 bg-white dark:bg-dark-card p-6 rounded-lg shadow-lg chart-container"><h3 class="font-bold mb-4">Projects per Location</h3><div class="h-80"><canvas id="locationChart"></canvas></div></div>
        `;
        
        // --- THIS IS THE FULL CHART CREATION LOGIC THAT WAS MISSING ---
        const chartConfigs = {
            statusChart: { type: 'doughnut', data: { labels: Object.keys(data.status), datasets: [{ data: Object.values(data.status) }] } },
            sectorChart: { type: 'pie', data: { labels: Object.keys(data.sector), datasets: [{ data: Object.values(data.sector) }] } },
            timelinessChart: { type: 'pie', data: { labels: Object.keys(data.timeliness), datasets: [{ data: Object.values(data.timeliness), backgroundColor: ['#16A34A', '#DC2626'] }] } },
            issueStatusChart: { type: 'doughnut', data: { labels: Object.keys(data.issueStatus), datasets: [{ data: Object.values(data.issueStatus) }] } },
            locationChart: { type: 'bar', data: { labels: Object.keys(data.location), datasets: [{ label: 'Projects', data: Object.values(data.location) }] } },
            budgetUtilizationChart: { type: 'bar', data: { 
                labels: Object.keys(data.budgetUtilization),
                datasets: [
                    { label: 'Allocated', data: Object.values(data.budgetUtilization).map(d => d.allocated), backgroundColor: '#6D28D9' },
                    { label: 'Spent', data: Object.values(data.budgetUtilization).map(d => d.spent), backgroundColor: '#F97316' }
                ]
            }}
        };
        
        const colors = ['#1E3A8A', '#F97316', '#6D28D9', '#16A34A', '#EAB308', '#DC2626', '#4B5563'];
        Chart.defaults.color = document.documentElement.classList.contains('dark') ? '#cbd5e1' : '#374151';

        for (const chartId in chartConfigs) {
            const canvas = document.getElementById(chartId);
            if (!canvas) continue;

            const config = chartConfigs[chartId];
            if (!config.data.datasets[0].backgroundColor) {
                config.data.datasets[0].backgroundColor = colors;
            }

            const options = { responsive: true, maintainAspectRatio: false, plugins: { legend: { position: 'right', labels: { color: Chart.defaults.color } } } };
            if (['locationChart', 'budgetUtilizationChart'].includes(chartId)) {
                options.plugins.legend = { display: true, position: 'top', labels: { color: Chart.defaults.color } };
                options.scales = { 
                    x: { stacked: chartId === 'budgetUtilizationChart', ticks: { color: Chart.defaults.color } }, 
                    y: { stacked: chartId === 'budgetUtilizationChart', beginAtZero: true, ticks: { color: Chart.defaults.color } } 
                };
            }
            charts[chartId] = new Chart(canvas, { ...config, options });
        }
    }

    // --- Data Fetching and Event Handling ---
    function fetchDashboardData() {
        loadingOverlay.classList.remove('hidden');
        const formData = new FormData(form);
        const selectedConstituencyText = constituencySelect.value ? constituencySelect.options[constituencySelect.selectedIndex].text : 'The Entire County';
        formData.append('report_title', selectedConstituencyText);
        const params = new URLSearchParams(formData);

        fetch('api/dashboard_data_api.php?' + params.toString())
            .then(res => res.ok ? res.json() : Promise.reject(res))
            .then(data => {
                if(data.error) throw new Error(data.error);
                updateDashboard(data);
            })
            .catch(err => {
                console.error("Dashboard Fetch Error:", err);
                alert('Failed to load dashboard data. Please check the console for errors.');
            })
            .finally(() => loadingOverlay.classList.add('hidden'));
    }

    form.addEventListener('submit', e => { e.preventDefault(); fetchDashboardData(); });

    document.getElementById('reset-filters').addEventListener('click', () => {
        form.reset();
        wardSelect.innerHTML = '<option value="">All Wards</option>';
        wardSelect.disabled = true;
        fetchDashboardData();
    });

    // --- PDF Export Logic ---
    const pdfModal = document.getElementById('pdf-preview-modal');
    const pdfPreviewBody = document.getElementById('pdf-preview-body');
    
    document.getElementById('open-pdf-preview').addEventListener('click', () => {
        const contentToPrint = document.getElementById('dashboard-content').cloneNode(true);
        
        // Convert canvas charts to static images for printing
        contentToPrint.querySelectorAll('canvas').forEach(canvas => {
            const chartId = canvas.id;
            if (charts[chartId]) {
                const image = new Image();
                image.src = charts[chartId].toBase64Image();
                image.style.width = '100%';
                image.style.height = 'auto';
                image.className = 'rounded-md shadow';
                canvas.parentNode.replaceChild(image, canvas);
            }
        });

        const header = `<div class="pb-4 mb-4 border-b-2 border-gray-200"><div class="flex items-center justify-between"><div class="flex items-center space-x-3"><img src="admin/assets/hbc_logo.png" class="h-12" alt="Logo"><div><span class="font-extrabold text-xl text-brand-blue">GENOWA</span><span class="block font-semibold text-xs text-brand-orange -mt-1">DELIVERY UNIT</span></div></div><div class="text-right"><p class="font-bold text-lg">Dashboard Report</p><p class="text-sm text-gray-600">${currentReportTitle}</p><p class="text-sm text-gray-600">Generated: ${new Date().toLocaleDateString()}</p></div></div></div>`;
        
        pdfPreviewBody.innerHTML = header;
        pdfPreviewBody.appendChild(contentToPrint);
        pdfModal.classList.remove('hidden');
    });

    document.getElementById('close-pdf-preview').addEventListener('click', () => pdfModal.classList.add('hidden'));

    document.getElementById('confirm-pdf-download').addEventListener('click', () => {
        const element = document.getElementById('pdf-preview-body');
        const opt = {
            margin:       [0, 0.5, 0.8, 0.5],
            filename:     'genowa_dashboard_report.pdf',
            // UPDATED: This setting respects the CSS grid and `page-break-inside: avoid`
            // allowing charts to flow naturally while preventing them from being cut.
            pagebreak:    { mode: ['css', 'avoid-all'] },
            image:        { type: 'jpeg', quality: 0.98 },
            html2canvas:  { scale: 2, useCORS: true },
            jsPDF:        { unit: 'in', format: 'letter', orientation: 'portrait' }
        };
        
        html2pdf().from(element).set(opt).toPdf().get('pdf').then(function (pdf) {
            const totalPages = pdf.internal.getNumberOfPages();
            const pageWidth = pdf.internal.pageSize.getWidth();
            const pageHeight = pdf.internal.pageSize.getHeight();

            for (let i = 1; i <= totalPages; i++) {
                pdf.setPage(i);

                // --- Draw Footer Divider Line ---
                pdf.setFillColor(249, 115, 22); // Brand Orange color
                pdf.rect(opt.margin[1], pageHeight - 0.65, pageWidth - opt.margin[1] - opt.margin[3], 0.02, 'F');

                // --- Draw Footer Text ---
                pdf.setFontSize(9);
                pdf.setTextColor(100); // Gray color for text
                
                // Add "Genowa Delivery Unit" to the left
                pdf.text('Genowa Delivery Unit', opt.margin[1], pageHeight - 0.5);

                // Add "Page X of Y" to the right
                const pageText = `Page ${i} of ${totalPages}`;
                const pageTextWidth = pdf.getStringUnitWidth(pageText) * pdf.internal.getFontSize() / pdf.internal.scaleFactor;
                pdf.text(pageText, pageWidth - pageTextWidth - opt.margin[3], pageHeight - 0.5);
            }
        }).save();
        
        pdfModal.classList.add('hidden');
    });

    fetchDashboardData(); // Initial load
});
</script>

<?php include 'includes/footer.php'; ?>