<?php
// FILE: public_html/contact.php
// PURPOSE: "Contact Us / Report an Issue" page with form and processing logic.

require_once 'db_connect.php';
require_once 'admin/functions.php';

$errors = [];
$success_message = '';
$inputs = [];

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitize and store inputs to re-populate the form on error
    $inputs['name'] = trim($_POST['name']);
    $inputs['email'] = trim($_POST['email']);
    $inputs['phone_number'] = trim($_POST['phone_number']);
    $inputs['subject'] = trim($_POST['subject']);
    $inputs['message'] = trim($_POST['message']);
    
    // --- Validation ---
    if (empty($inputs['name'])) $errors['name'] = "Your name is required.";
    if (empty($inputs['subject'])) $errors['subject'] = "A subject is required.";
    if (empty($inputs['message'])) $errors['message'] = "A message is required.";

    // --- UPDATED: Separate validation for Email and Phone Number ---
    if (empty($inputs['email'])) {
        $errors['email'] = "Your email address is required.";
    } elseif (!filter_var($inputs['email'], FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = "Please enter a valid email address.";
    }
    
    if (empty($inputs['phone_number'])) {
        $errors['phone_number'] = "Your phone number is required.";
    } elseif (!preg_match('/^(?:\+254|0)?(7|1)\d{8}$/', $inputs['phone_number'])) {
        $errors['phone_number'] = "Please enter a valid Kenyan phone number (e.g., 0712345678).";
    }


    // --- Process submission if no errors ---
    if (empty($errors)) {
        try {
            // To avoid changing the DB schema, we'll combine email and phone for the `contact_info` column
            $contact_info_combined = "Email: " . $inputs['email'] . " | Phone: " . $inputs['phone_number'];
            
            // Prepare SQL to insert new message
            $sql = "INSERT INTO issues_and_contacts (name, contact_info, subject, message) VALUES (:name, :contact_info, :subject, :message)";
            $stmt = $pdo->prepare($sql);
            
            $stmt->execute([
                'name' => $inputs['name'],
                'contact_info' => $contact_info_combined,
                'subject' => $inputs['subject'],
                'message' => $inputs['message']
            ]);

            // --- START: Send Notifications ---

            // 1. Send an email notification to the admin
            $admin_email = "projects@genowa.co.ke";
            $email_subject_to_admin = "New Genowa Contact: " . $inputs['subject'];
            $email_body_to_admin = "
                <p>You have a new submission from the Genowa Delivery Unit contact form.</p>
                <p><strong>Name:</strong> " . e($inputs['name']) . "</p>
                <p><strong>Email:</strong> " . e($inputs['email']) . "</p>
                <p><strong>Phone:</strong> " . e($inputs['phone_number']) . "</p>
                <p><strong>Subject:</strong> " . e($inputs['subject']) . "</p>
                <hr>
                <p><strong>Message:</strong></p>
                <p>" . nl2br(e($inputs['message'])) . "</p>
            ";
            send_email($admin_email, $email_subject_to_admin, $email_body_to_admin);

            // 2. Send notifications to the user who submitted the form
            $confirmation_message_email = "<p>Dear " . e($inputs['name']) . ",<br><br>Thank you for contacting the Genowa Delivery Unit. Your message regarding '" . e($inputs['subject']) . "' has been received. Our team will get back to you shortly.</p>";
            $confirmation_message_sms = "Dear " . e($inputs['name']) . ", we've received your message regarding '" . e($inputs['subject']) . "'. Thank you for contacting the Genowa Delivery Unit.";

            // Send both email and SMS confirmations
            send_email($inputs['email'], "Message Received - Genowa Delivery Unit", $confirmation_message_email);
            send_sms($inputs['phone_number'], $confirmation_message_sms);
            
            // --- END: Send Notifications ---


            $success_message = "Thank you for contacting us! Your message has been received, and our team will get back to you shortly.";
            $inputs = []; // Clear form on success

        } catch (PDOException $e) {
            $errors['db'] = "An error occurred. Please try again later.";
        }
    }
}

include 'includes/header.php';
?>

<div class="bg-brand-blue">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-16 text-center">
        <h1 class="text-4xl md:text-5xl font-extrabold text-white">Get In Touch</h1>
        <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-300">
            Have a question, feedback, or wish to report an issue with a project? We'd love to hear from you.
        </p>
    </div>
</div>

<main class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
    <div class="max-w-xl mx-auto">
        <div class="bg-white rounded-lg shadow-xl p-8">
             <?php if ($success_message): ?>
                <div class="text-center p-6 bg-green-50 border-l-4 border-green-500">
                    <h3 class="text-lg font-bold text-green-800">Message Sent Successfully!</h3>
                    <p class="mt-2 text-gray-600"><?php echo e($success_message); ?></p>
                </div>
            <?php else: ?>
                <form action="contact.php" method="post" class="space-y-6">
                    <?php if (!empty($errors)): ?>
                        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                            <p class="font-bold">Please correct the errors below:</p>
                            <ul class="list-disc list-inside mt-2">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo e($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <!-- UPDATED: Restyled Form Fields with Icons -->
                    <div>
                        <label for="name" class="block text-sm font-medium text-gray-700">Full Name</label>
                        <div class="relative mt-1 rounded-md shadow-sm">
                            <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z" /></svg></div>
                            <input type="text" name="name" id="name" value="<?php echo e($inputs['name'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3">
                        </div>
                    </div>
                     <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                           <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                           <div class="relative mt-1 rounded-md shadow-sm">
                               <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M21.75 6.75v10.5a2.25 2.25 0 01-2.25 2.25h-15a2.25 2.25 0 01-2.25-2.25V6.75m19.5 0A2.25 2.25 0 0019.5 4.5h-15a2.25 2.25 0 00-2.25 2.25m19.5 0v.243a2.25 2.25 0 01-1.07 1.916l-7.5 4.615a2.25 2.25 0 01-2.36 0L3.32 8.91a2.25 2.25 0 01-1.07-1.916V6.75" /></svg></div>
                               <input type="email" name="email" id="email" value="<?php echo e($inputs['email'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3">
                           </div>
                        </div>
                         <div>
                           <label for="phone_number" class="block text-sm font-medium text-gray-700">Phone Number</label>
                            <div class="relative mt-1 rounded-md shadow-sm">
                               <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M2.25 6.75c0 8.284 6.716 15 15 15h2.25a2.25 2.25 0 002.25-2.25v-1.372c0-.516-.351-.966-.852-1.091l-4.423-1.106c-.44-.11-.902.055-1.173.417l-.97 1.293c-.282.376-.769.542-1.21.38a12.035 12.035 0 01-7.143-7.143c-.162-.441.004-.928.38-1.21l1.293-.97c.363-.271.527-.734.417-1.173L6.963 3.102a1.125 1.125 0 00-1.091-.852H4.5A2.25 2.25 0 002.25 4.5v2.25z" /></svg></div>
                               <input type="tel" name="phone_number" id="phone_number" value="<?php echo e($inputs['phone_number'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3" placeholder="0712345678">
                            </div>
                        </div>
                    </div>
                     <div>
                        <label for="subject" class="block text-sm font-medium text-gray-700">Subject</label>
                        <div class="relative mt-1 rounded-md shadow-sm">
                           <div class="pointer-events-none absolute inset-y-0 left-0 flex items-center pl-3"><svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M7.5 8.25h9m-9 3H12m-9.75 1.51c0 1.6 1.123 2.994 2.707 3.227 1.129.166 2.27.293 3.423.379.35.026.67.21.865.501L12 21l2.755-4.133a1.14 1.14 0 01.865-.501 48.17 48.17 0 003.423-.379c1.584-.233 2.707-1.626 2.707-3.228V6.741c0-1.602-1.123-2.995-2.707-3.228A48.394 48.394 0 0012 3c-2.392 0-4.744.175-7.043.513C3.373 3.746 2.25 5.14 2.25 6.741v6.018z" /></svg></div>
                           <input type="text" name="subject" id="subject" value="<?php echo e($inputs['subject'] ?? ''); ?>" required class="block w-full rounded-md border-gray-300 pl-10 focus:border-brand-purple focus:ring-brand-purple sm:text-sm py-3">
                        </div>
                    </div>
                    <div>
                        <label for="message" class="block text-sm font-medium text-gray-700">Your Message</label>
                        <textarea name="message" id="message" rows="5" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3 text-base"><?php echo e($inputs['message'] ?? ''); ?></textarea>
                    </div>
                    <div class="pt-2">
                        <button type="submit" class="w-full inline-flex justify-center rounded-md border border-transparent bg-brand-orange py-3 px-6 text-base font-medium text-white shadow-sm hover:bg-opacity-90 transition-transform duration-200 hover:scale-105">
                            Send Message
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</main>

<?php include 'includes/footer.php'; ?>
