<?php
// Set the timezone to get the correct local time
date_default_timezone_set('Africa/Nairobi');
$hour = date('G'); // Get the current hour in 24-hour format (0-23)
$greeting = '';

// Determine the appropriate greeting based on the time of day
if ($hour >= 5 && $hour < 12) {
    $greeting = "Good morning";
} elseif ($hour >= 12 && $hour < 18) {
    $greeting = "Good afternoon";
} else {
    $greeting = "Good evening";
}

// Construct the full, personalized greeting message
$full_greeting = "$greeting! I am Genowa Bot, ready to assist you. How may I help you today?";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Genowa Bot | Civic-Tech Assistant</title>
    
    <!-- Enterprise Upgrade: Using Tailwind CSS for a modern, maintainable UI -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- Include the 'marked' library to render Markdown from the bot -->
    <script src="https://cdn.jsdelivr.net/npm/marked/marked.min.js"></script>
    
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Using a minimal style block for things Tailwind doesn't handle out-of-the-box */
        body {
            font-family: 'Inter', sans-serif;
        }
        /* Custom scrollbar for a cleaner look */
        #chat-box::-webkit-scrollbar {
            width: 6px;
        }
        #chat-box::-webkit-scrollbar-track {
            background: #f1f5f9;
        }
        #chat-box::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 3px;
        }
        #chat-box::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }
        /* Style for the bot's Markdown content */
        .bot-content ul {
            list-style-type: disc;
            padding-left: 20px;
            margin-top: 8px;
        }
        .bot-content li {
            margin-bottom: 4px;
        }
        .bot-content strong {
            font-weight: 600;
            color: #1e40af; /* A darker blue for emphasis */
        }
    </style>
</head>
<body class="bg-slate-100 flex items-center justify-center min-h-screen">
    <div id="chat-container" class="w-full max-w-2xl h-[95vh] max-h-[800px] bg-white rounded-2xl shadow-2xl flex flex-col overflow-hidden border border-slate-200">
        
        <!-- Header -->
        <div id="chat-header" class="bg-blue-700 text-white p-4 sm:p-5 flex items-center justify-between shadow-md">
            <div class="flex items-center gap-4">
                <img src="/admin/assets/genowad.png" alt="Genowa Bot" class="w-12 h-12 rounded-full border-2 border-white/50">
                <div>
                    <h1 class="text-xl font-bold">Genowa Bot</h1>
                    <p class="text-sm text-blue-200">Your Civic-Tech Assistant</p>
                </div>
            </div>
            <div id="lang-selector-container">
                <select id="lang-selector" aria-label="Select language for voice input" class="bg-white/20 border border-white/30 rounded-md px-2 py-1 text-xs text-white focus:outline-none focus:ring-2 focus:ring-white">
                    <option value="en-US" class="text-black">English</option>
                    <option value="sw-KE" class="text-black">Kiswahili</option>
                    <option value="luo-KE" class="text-black">Dholuo</option>
                </select>
            </div>
        </div>

        <!-- Chat Box -->
        <div id="chat-box" aria-live="polite" class="flex-grow p-4 sm:p-6 overflow-y-auto bg-slate-50">
            <!-- Greeting message will be injected here by JavaScript -->
        </div>

        <!-- Input Form -->
        <form id="chat-form" autocomplete="off" class="p-4 bg-white border-t border-slate-200 flex items-center gap-3">
            <input type="text" id="user-input" placeholder="Ask about county services..." maxlength="400" required aria-label="Type your message here" class="flex-grow w-full px-4 py-3 border border-slate-300 rounded-full focus:outline-none focus:ring-2 focus:ring-blue-500 transition-shadow">
            <button id="mic-btn" type="button" class="p-3 rounded-full text-slate-500 hover:bg-blue-100 hover:text-blue-600 focus:outline-none focus:ring-2 focus:ring-blue-500 transition-colors" aria-label="Use microphone">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"></path><path d="M19 10v2a7 7 0 0 1-14 0v-2"></path><line x1="12" y1="19" x2="12" y2="23"></line></svg>
            </button>
            <button id="send-btn" type="submit" class="p-3 rounded-full bg-blue-600 text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-transform active:scale-95" aria-label="Send Message">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="22" y1="2" x2="11" y2="13"></line><polygon points="22 2 15 22 11 13 2 9 22 2"></polygon></svg>
            </button>
        </form>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const chatForm = document.getElementById('chat-form');
            const userInput = document.getElementById('user-input');
            const chatBox = document.getElementById('chat-box');
            const micBtn = document.getElementById('mic-btn');
            const langSelector = document.getElementById('lang-selector');

            // Enterprise Upgrade: Store conversation history to send to the backend
            let conversationHistory = [];
            const MAX_HISTORY = 6; // Keep the last 3 user/bot message pairs

            // --- Initial Greeting ---
            appendMessage(`<?php echo htmlspecialchars($full_greeting, ENT_QUOTES); ?>`, 'bot');

            // --- Form Submission Handler ---
            chatForm.addEventListener('submit', async (e) => {
                e.preventDefault();
                const userMessage = userInput.value.trim();
                if (userMessage === '') return;

                appendMessage(userMessage, 'user');
                addToHistory('user', userMessage);
                userInput.value = '';
                userInput.focus();
                
                // NEW: Multi-stage loading indicator
                const loadingIndicator = showLoadingIndicator();

                try {
                    const response = await fetch('chat_handler.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            message: userMessage,
                            language: langSelector.value,
                            history: conversationHistory // Send history with the request
                        }),
                    });

                    if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
                    
                    const data = await response.json();
                    
                    // Replace loading indicator with the final message
                    removeLoadingIndicator(loadingIndicator, data.reply);
                    addToHistory('assistant', data.reply);

                } catch (error) {
                    console.error('Error:', error);
                    removeLoadingIndicator(loadingIndicator, 'Sorry, there was an error connecting to our servers. Please try again later.');
                }
            });

            /**
             * Appends a message to the chat box.
             * @param {string} message - The text content of the message.
             * @param {'user' | 'bot'} role - The role of the message sender.
             */
            function appendMessage(message, role) {
                const messageElement = document.createElement('div');
                messageElement.className = `w-full flex mb-4 animate-fade-in ${role === 'user' ? 'justify-end' : 'justify-start'}`;

                // NEW: Using img tag for the bot avatar
                const avatarSrc = role === 'user' 
                    ? `<div class="w-10 h-10 rounded-full bg-slate-200 flex items-center justify-center text-slate-500 font-semibold">You</div>`
                    : `<img src="/admin/assets/genowad.png" alt="Genowa Bot" class="w-10 h-10 rounded-full shadow-md">`;

                const messageContent = document.createElement('div');
                messageContent.className = `p-3 rounded-2xl max-w-lg ${role === 'user' ? 'bg-blue-600 text-white rounded-br-none' : 'bg-slate-200 text-slate-800 rounded-bl-none'}`;
                
                // Use marked to render bot responses as HTML, and textContent for user messages to prevent XSS
                if (role === 'bot') {
                    messageContent.innerHTML = `<div class="bot-content">${marked.parse(message)}</div>`;
                } else {
                    messageContent.textContent = message;
                }
                
                const messageContainer = document.createElement('div');
                messageContainer.className = `flex items-end gap-2 ${role === 'user' ? 'flex-row-reverse' : ''}`;
                messageContainer.innerHTML = avatarSrc;
                messageContainer.appendChild(messageContent);

                messageElement.appendChild(messageContainer);
                chatBox.appendChild(messageElement);
                chatBox.scrollTop = chatBox.scrollHeight;
            }

            /**
             * NEW: Shows a multi-stage loading indicator and returns the element.
             * @returns {HTMLElement} The loading indicator element.
             */
            function showLoadingIndicator() {
                const loadingElement = document.createElement('div');
                loadingElement.className = 'w-full flex mb-4 animate-fade-in justify-start';
                loadingElement.id = 'loading-indicator';

                const content = document.createElement('div');
                content.className = 'p-3 rounded-2xl max-w-lg bg-slate-200 text-slate-500 rounded-bl-none italic';
                content.textContent = 'Thinking...';

                const container = document.createElement('div');
                container.className = 'flex items-end gap-2';
                container.innerHTML = `<img src="/admin/assets/genowad.png" alt="Genowa Bot" class="w-10 h-10 rounded-full shadow-md">`;
                container.appendChild(content);
                loadingElement.appendChild(container);
                chatBox.appendChild(loadingElement);
                chatBox.scrollTop = chatBox.scrollHeight;

                // Cycle through different loading messages
                const stages = ['Searching knowledge base...', 'Compiling the best answer...'];
                let stageIndex = 0;
                const intervalId = setInterval(() => {
                    if (stageIndex < stages.length) {
                        content.textContent = stages[stageIndex];
                        stageIndex++;
                    } else {
                        clearInterval(intervalId);
                    }
                }, 2000); // Change message every 2 seconds

                // Store interval to clear it when the response arrives
                loadingElement.dataset.intervalId = intervalId;
                return loadingElement;
            }

            /**
             * NEW: Removes the loading indicator and replaces it with the final bot message.
             * @param {HTMLElement} indicator - The loading indicator element to remove.
             * @param {string} finalMessage - The bot's final reply.
             */
            function removeLoadingIndicator(indicator, finalMessage) {
                if (!indicator) return;
                clearInterval(indicator.dataset.intervalId); // Stop the text cycling
                
                // Create a proper bot message to replace the indicator
                const messageElement = document.createElement('div');
                messageElement.className = `w-full flex mb-4 animate-fade-in justify-start`;

                const messageContent = document.createElement('div');
                messageContent.className = `p-3 rounded-2xl max-w-lg bg-slate-200 text-slate-800 rounded-bl-none`;
                messageContent.innerHTML = `<div class="bot-content">${marked.parse(finalMessage)}</div>`;
                
                const messageContainer = document.createElement('div');
                messageContainer.className = `flex items-end gap-2`;
                messageContainer.innerHTML = `<img src="/admin/assets/genowad.png" alt="Genowa Bot" class="w-10 h-10 rounded-full shadow-md">`;
                messageContainer.appendChild(messageContent);
                messageElement.appendChild(messageContainer);

                // Replace the indicator with the final message
                indicator.replaceWith(messageElement);
                chatBox.scrollTop = chatBox.scrollHeight;
            }
            
            /**
             * Enterprise Upgrade: Adds a message to the conversation history.
             * @param {'user' | 'assistant'} role - The role of the message sender.
             * @param {string} content - The message text.
             */
            function addToHistory(role, content) {
                conversationHistory.push({ role, content });
                // Keep the history from getting too long
                if (conversationHistory.length > MAX_HISTORY) {
                    conversationHistory = conversationHistory.slice(-MAX_HISTORY);
                }
            }

            // --- Speech Recognition Setup (No changes needed here, but kept for completeness) ---
            const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
            if (SpeechRecognition) {
                const recognition = new SpeechRecognition();
                recognition.continuous = false;
                recognition.interimResults = false;
                micBtn.addEventListener('click', () => {
                    micBtn.classList.contains('recording') ? recognition.stop() : recognition.start();
                });
                recognition.onstart = () => {
                    micBtn.classList.add('recording', 'bg-red-500', 'text-white');
                    userInput.placeholder = 'Listening...';
                };
                recognition.onend = () => {
                    micBtn.classList.remove('recording', 'bg-red-500', 'text-white');
                    userInput.placeholder = 'Ask about county services...';
                };
                recognition.onresult = (event) => {
                    userInput.value = event.results[0][0].transcript;
                    chatForm.requestSubmit();
                };
                recognition.onerror = (event) => console.error('Speech recognition error:', event.error);
            } else {
                micBtn.style.display = 'none';
            }
        });
    </script>
</body>
</html>
