<?php
// Enable error reporting for debugging (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include required files
require_once 'includes/db.php';
require_once 'includes/functions.php';
require_once 'includes/functions2.php';

session_start();

// CSRF Token Validation
if (!isset($_POST['token']) || $_POST['token'] !== $_SESSION['token']) {
    die('<div style="color:red; text-align:center; margin-top:40px;">Invalid CSRF token. Please reload the form and try again.</div>');
}

// Clean & Validate Inputs
function clean_input($data) {
    return htmlspecialchars(trim($data));
}

$name            = clean_input($_POST['name'] ?? '');
$email           = clean_input($_POST['email'] ?? '');
$password        = $_POST['password'] ?? '';
$phone_number    = clean_input($_POST['phone_number'] ?? '');
$payment_number  = clean_input($_POST['payment_number'] ?? '');
$location        = clean_input($_POST['location'] ?? '');
$bio             = clean_input($_POST['bio'] ?? '');
$expertise       = clean_input($_POST['expertise'] ?? '');

// Validation Errors
$errors = [];

// Validate Name (only letters, spaces, apostrophes)
if (empty($name) || !preg_match("/^[a-zA-Z\s'.-]+$/", $name)) {
    $errors[] = "Full name should contain only letters, spaces, apostrophes or periods.";
}

// Validate Email
if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $errors[] = "Please enter a valid email address.";
}

// Validate Password
if (empty($password) || strlen($password) < 6) {
    $errors[] = "Password must be at least 6 characters long.";
}

// Validate Phone Number (must start with 07 or 01 and be 10 digits)
if (empty($phone_number) || !preg_match("/^(07|01)\d{8}$/", $phone_number)) {
    $errors[] = "Phone number must be 10 digits and start with 07 or 01.";
}

// Validate Location
if (empty($location)) {
    $errors[] = "Location is required.";
}

// If there are validation errors, show them
if (!empty($errors)) {
    echo '<div style="max-width:600px; margin:40px auto; background:#fff3f3; border:1px solid #ffb3b3; border-radius:10px; padding:20px; font-family:sans-serif;">';
    echo '<h3 style="color:#d32f2f;">There were some problems with your submission:</h3><ul style="color:#555;">';
    foreach ($errors as $error) {
        echo "<li>• $error</li>";
    }
    echo '</ul><p><a href="register.php" style="color:#1976d2; text-decoration:underline;">Go back and correct them</a></p></div>';
    exit;
}

// Check if email already exists
try {
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->rowCount() > 0) {
        die('<div style="color:red; text-align:center; margin-top:40px;">This email is already registered. <a href="register.php" style="color:blue;">Go back</a></div>');
    }

    // Hash password securely
    $hashed_password = password_hash($password, PASSWORD_BCRYPT);

    // Default values
    $role = 'blogger';
    $status = 'pending';
    $supervisor_id = null;

    // Insert into database
    $insert = $pdo->prepare("
        INSERT INTO users 
        (email, password, name, location, phone_number, payment_number, bio, expertise, role, status, supervisor_id, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $insert->execute([
        $email,
        $hashed_password,
        $name,
        $location,
        $phone_number,
        $payment_number,
        $bio,
        $expertise,
        $role,
        $status,
        $supervisor_id
    ]);

    // Clear token after success
    unset($_SESSION['token']);

    echo '
    <div style="max-width:600px; margin:60px auto; background:#e8f5e9; border:1px solid #a5d6a7; border-radius:10px; padding:30px; font-family:sans-serif; text-align:center;">
      <h2 style="color:#2e7d32;">✅ Registration Successful!</h2>
      <p style="color:#333;">Thank you, <strong>' . htmlspecialchars($name) . '</strong>, for joining the Genowa Movement.</p>
      <p style="margin-top:10px;">Your account is currently under review. You will be notified via email once approved.</p>
      <a href="register.php" style="display:inline-block; margin-top:20px; background:#2e7d32; color:white; padding:10px 20px; border-radius:8px; text-decoration:none;">Register Another</a>
    </div>';
    
} catch (PDOException $e) {
    echo '<div style="color:red; text-align:center; margin-top:40px;">Database error: ' . htmlspecialchars($e->getMessage()) . '</div>';
    exit;
}
?>
