<?php
// FILE: pages/dashboard.php
// This page is the main dashboard and dynamically displays content based on user role.

// Data for all roles is fetched here, and the HTML below will
// decide which sections to show based on $user_role.

// --- Data Fetching ---

// For Bloggers:
$blogger_stats = getBloggerStats($user_id);
$blogger_posts = getBloggerRecentPosts($user_id);

// For Supervisors:
$supervisor_stats = getSupervisorStats($user_id);
$supervisor_team = getSupervisorTeamDetails($user_id);

// For Admins:
$admin_stats = getAdminStats();
$admin_supervisors = getAdminSupervisorOverview();

?>

<!-- 
    This page uses Alpine.js (loaded in header.php) for its tabs.
    The main container's x-data is set dynamically based on the user's role.
-->
<?php
    // Set the default active tab
    $default_tab = 'my_stats'; // Default for Bloggers
    if ($user_role === 'supervisor') $default_tab = 'team_management';
    if ($user_role === 'admin') $default_tab = 'admin_overview';
?>

<div x-data="{ activeTab: '<?php echo $default_tab; ?>' }">

    <!-- Dashboard Header -->
    <h1 class="text-3xl font-extrabold text-gray-900 mb-6">Dashboard</h1>

    <!-- Role-Based Tab Navigation -->
    <div class="mb-6 border-b border-gray-200">
        <nav class="-mb-px flex space-x-6 overflow-x-auto" aria-label="Tabs">
            
            <!-- Blogger Tabs -->
            <?php if ($user_role === 'blogger'): ?>
                <button @click="activeTab = 'my_stats'"
                        :class="activeTab === 'my_stats' ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                        class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200">
                    My Stats
                </button>
            <?php endif; ?>

            <!-- Supervisor Tabs -->
            <?php if ($user_role === 'supervisor' || $user_role === 'admin'): ?>
                <button @click="activeTab = 'my_stats'"
                        :class="activeTab === 'my_stats' ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                        class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200">
                    My Stats
                </button>
                <button @click="activeTab = 'team_management'"
                        :class="activeTab === 'team_management' ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                        class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200">
                    Team Management
                </button>
            <?php endif; ?>

            <!-- Admin Tabs -->
            <?php if ($user_role === 'admin'): ?>
                <button @click="activeTab = 'admin_overview'"
                        :class="activeTab === 'admin_overview' ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'"
                        class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-200">
                    Admin Overview
                </button>
            <?php endif; ?>
        </nav>
    </div>

    <!-- 
    =================================
    | Tab Panels
    =================================
    -->

    <!-- 
    =================================
    | PANEL: MY STATS (All Users)
    =================================
    -->
    <div x-show="activeTab === 'my_stats'" x-cloak>
        <h2 class="text-2xl font-semibold text-gray-800 mb-4">My Stats</h2>
        
        <!-- Blogger/Supervisor Personal Stats -->
        <?php if ($user_role === 'blogger' || $user_role === 'supervisor'): ?>
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <!-- Card: Total Earnings -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">My Total Earnings</span>
                        <i class="fas fa-coins text-xl text-yellow-400"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        KES <?php echo number_format($blogger_stats['earnings'], 2); ?>
                    </p>
                </div>
                <!-- Card: Total Reach -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">My Total Reach</span>
                        <i class="fas fa-bullhorn text-xl text-blue-500"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($blogger_stats['reach']); ?>
                    </p>
                </div>
                <!-- Card: Approved Posts -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">My Completed Posts</span>
                        <i class="fas fa-check-circle text-xl text-green-500"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($blogger_stats['posts']); ?>
                    </p>
                </div>
            </div>
            
            <!-- Blogger: Recent Posts Table -->
            <?php if ($user_role === 'blogger'): ?>
                <h3 class="text-xl font-semibold text-gray-800 mb-4">My Recent Posts</h3>
                <div class="bg-white rounded-xl shadow-lg overflow-hidden border border-gray-100">
                    <!-- Responsive Table Container -->
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Post URL</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Submitted</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (empty($blogger_posts)): ?>
                                    <tr>
                                        <td colspan="3" class="px-6 py-8 text-center text-gray-500">
                                            You have not submitted any posts yet.
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($blogger_posts as $post): ?>
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="text-sm text-gray-900 truncate" style="max-width: 300px;"><?php echo htmlspecialchars($post['url']); ?></div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="text-sm text-gray-900"><?php echo date('M d, Y', strtotime($post['submittedAt'])); ?></div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <?php
                                                    // This is the FIX for the blank status bug
                                                    $status_label = 'Unknown';
                                                    $status_class = 'bg-gray-100 text-gray-800'; // Default
                                                    
                                                    switch ($post['status']) {
                                                        case 'pending':
                                                            $status_label = 'Pending';
                                                            $status_class = 'bg-yellow-100 text-yellow-800';
                                                            break;
                                                        case 'performance_review': // CHANGED
                                                            $status_label = 'Approved'; // Was "Under Review"
                                                            $status_class = 'bg-blue-100 text-blue-800'; // Blue = approved, but pending 2nd review
                                                            break;
                                                        case 'completed': // CHANGED
                                                            $status_label = 'Completed'; // Was "Approved"
                                                            $status_class = 'bg-green-100 text-green-800'; // Green = final
                                                            break;
                                                        case 'rejected':
                                                            $status_label = 'Rejected';
                                                            $status_class = 'bg-red-100 text-red-800';
                                                            break;
                                                    }
                                                ?>
                                                <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo $status_class; ?>">
                                                    <?php echo $status_label; ?>
                                                </span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endif; ?>

        <?php else: ?>
            <p class="text-gray-600">You are logged in as an Admin. Your personal stats are not tracked. Use the tabs to manage supervisors and teams.</p>
        <?php endif; ?>
    </div>

    <!-- 
    =================================
    | PANEL: TEAM MANAGEMENT (Supervisor/Admin)
    =================================
    -->
    <?php if ($user_role === 'supervisor' || $user_role === 'admin'): ?>
        <div x-show="activeTab === 'team_management'" x-cloak>
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">My Team Management</h2>
            
            <!-- Supervisor Team Stats -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <!-- Card: Team Commissions -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">My Total Commissions</span>
                        <i class="fas fa-percent text-xl text-purple-500"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        KES <?php echo number_format($supervisor_stats['commissions'], 2); ?>
                    </p>
                </div>
                <!-- Card: Team Reach -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">Team Total Reach</span>
                        <i class="fas fa-users text-xl text-teal-500"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($supervisor_stats['teamReach']); ?>
                    </p>
                </div>
                <!-- Card: Team Approved Posts -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-500">Team Completed Posts</span>
                        <i class="fas fa-file-alt text-xl text-indigo-500"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($supervisor_stats['teamPosts']); ?>
                    </p>
                </div>
            </div>

            <!-- Team Monitoring Table -->
            <h3 class="text-xl font-semibold text-gray-800 mb-4">My Team Monitoring</h3>
            <div class="bg-white rounded-xl shadow-lg border border-gray-100">
                <!-- Responsive Table Container -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Blogger</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Completed Posts</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total Reach</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total Earnings</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($supervisor_team)): ?>
                                <tr>
                                    <td colspan="5" class="px-6 py-8 text-center text-gray-500">
                                        You have no bloggers on your team yet.
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($supervisor_team as $blogger): ?>
                                    <tr>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($blogger['email']); ?></div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <?php if ($blogger['status'] === 'pending'): ?>
                                                <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                                    Pending Approval
                                                </span>
                                            <?php else: ?>
                                                <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                                    Approved
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700"><?php echo number_format($blogger['posts']); ?></td>
                                        <td class="px-6 py-4 whitespace-nowMrap text-sm text-gray-700"><?php echo number_format($blogger['reach']); ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">KES <?php echo number_format($blogger['earnings'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- 
    =================================
    | PANEL: ADMIN OVERVIEW (Admin Only)
    =================================
    -->
    <?php if ($user_role === 'admin'): ?>
        <div x-show="activeTab === 'admin_overview'" x-cloak>
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">Site-Wide Admin Overview</h2>
            
            <!-- Admin Stats -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <!-- Card: Total Earnings -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <span class="text-sm font-medium text-gray-500">Total Earnings (All)</span>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        KES <?php echo number_format($admin_stats['totalEarnings'], 2); ?>
                    </p>
                </div>
                <!-- Card: Total Reach -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <span class="text-sm font-medium text-gray-500">Total Reach (All)</span>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($admin_stats['totalReach']); ?>
                    </all-users>
                </div>
                <!-- Card: Total Users -->
                <div class="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
                    <span class="text-sm font-medium text-gray-500">Total Approved Users</span>
                    <p class="mt-2 text-3xl font-extrabold text-gray-900">
                        <?php echo number_format($admin_stats['totalUsers']); ?>
                    </p>
                </div>
                <!-- Card: Manage Users -->
                <a href="index.php?page=manage_users" class="group bg-gray-800 rounded-xl shadow-lg p-6 border border-gray-700 transition-colors hover:bg-gray-700">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-300">Manage</span>
                        <i class="fas fa-users text-xl text-gray-400"></i>
                    </div>
                    <p class="mt-2 text-3xl font-extrabold text-white">
                        Manage Users
                    </p>
                </a>
            </div>

            <!-- Supervisor Overview Table -->
            <h3 class="text-xl font-semibold text-gray-800 mb-4">Supervisor Overview</h3>
            <div class="bg-white rounded-xl shadow-lg border border-gray-100">
                <!-- Responsive Table Container -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Supervisor</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Team Size</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Team Reach</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total Commissions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($admin_supervisors)): ?>
                                <tr>
                                    <td colspan="4" class="px-6 py-8 text-center text-gray-500">
                                        There are no supervisors in the system.
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($admin_supervisors as $supervisor): ?>
                                    <tr>
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($supervisor['email']); ?></div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700"><?php echo number_format($supervisor['teamSize']); ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700"><?php echo number_format($supervisor['teamReach']); ?></td>
                                        <td class_name="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">KES <?php echo number_format($supervisor['commissions'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    <?php endif; ?>

</div>

