<?php
// FILE: index.php
// This is the main controller for the logged-in area.

// 0. Set Timezone
// This ensures all date/time calculations are consistent
date_default_timezone_set('UTC');

// 1. Load Core Files
require_once 'includes/db.php';
require_once 'includes/functions.php';
require_once 'includes/functions2.php';

// 2. Start Session
secure_session_start();

// 3. Check Authentication & Status
// checkAuth() returns the user's status if they are logged in.
$user_status = checkAuth(); // This will redirect to login.php if not authenticated

// 4. Handle POST Actions
// This block processes form submissions *before* any HTML is sent.
$action = $_POST['action'] ?? '';
if ($_SERVER["REQUEST_METHOD"] == "POST" && !empty($action)) {
    
    // All action forms must have a CSRF token
    validateCSRF();
    
    switch ($action) {
        case 'submit_post':
            handleSubmitPost();
            header("Location: index.php?page=dashboard"); // Redirect to show new pending post
            exit;
        
        case 'approve_user':
            handleApproveUser();
            header("Location: index.php?page=approve_team");
            exit;
        
        case 'reject_user':
            handleRejectUser();
            header("Location: index.php?page=approve_team");
            exit;
        
        case 'approve_post':
            handleApprovePost();
            header("Location: index.php?page=review_posts");
            exit;
        
        case 'reject_post':
            handleRejectPost();
            header("Location: index.php?page=review_posts");
            exit;
            
        case 'perform_review':
            handlePerformanceReview();
            header("Location: index.php?page=review_performance");
            exit;
        
        case 'set_role':
            handleSetRole();
            header("Location: index.php?page=manage_users");
            exit;
            
        case 'update_profile':
            handleUpdateProfile();
            header("Location: index.php?page=my_profile");
            exit;
    }
}

// 5. Get Page & Handle Routing

// Get requested page, default to dashboard
$page = $_GET['page'] ?? 'dashboard';

// --- NEW PENDING USER ROUTING ---
// If user is logged in but NOT approved, force them to the profile page
if ($user_status !== 'approved' && $page !== 'my_profile' && $page !== 'logout') {
    // Allow them to go to 'my_profile' or 'logout', but nowhere else
    header("Location: index.php?page=my_profile");
    exit;
}
// If user IS approved and tries to see profile, that's fine.
// If user IS approved and lands on index, send to dashboard.
if ($user_status === 'approved' && $page === 'dashboard' && empty($_GET['page'])) {
     header("Location: index.php?page=dashboard");
     exit;
}


// 6. Whitelist Pages
// This is a security measure to prevent arbitrary file inclusion.
$allowed_pages = [
    'dashboard',
    'my_profile', // Add new profile page
    'submit_post',
    'review_posts',
    'review_performance',
    'approve_team',
    'manage_users',
    'feed'
];

// Allow profile page even for pending
if ($user_status !== 'approved') {
    $allowed_pages = ['my_profile']; // They can only see this page
}


if (!in_array($page, $allowed_pages)) {
    if ($user_status !== 'approved') {
        $page = 'my_profile'; // Force pending to profile
    } else {
        $page = 'dashboard'; // Default approved to dashboard
    }
}

// 7. Determine which page file to include
$page_file = "pages/{$page}.php";

// 8. Check if the page file exists
if (!file_exists($page_file)) {
    // This should not happen if whitelist is correct
    if ($user_status !== 'approved') {
        $page_file = "pages/my_profile.php";
    } else {
        $page_file = "pages/dashboard.php"; 
    }
    $page = str_replace(['pages/', '.php'], '', $page_file);
}

// 9. Load Header Template
// The header contains the sidebar, top bar, and all opening HTML tags.
require_once 'templates/header.php';

// 10. Show Success/Error Messages
// These are set in the session by the action handlers
if (isset($_SESSION['error_message'])) {
    echo '<div class="mb-6 p-4 bg-red-100 border-l-4 border-red-500 text-red-700 rounded-lg" role="alert">
            <p class="font-bold">Error</p>
            <p>' . htmlspecialchars($_SESSION['error_message']) . '</p>
          </div>';
    unset($_SESSION['error_message']);
}

if (isset($_SESSION['success_message'])) {
    echo '<div class="mb-6 p-4 bg-green-100 border-l-4 border-green-500 text-green-700 rounded-lg" role="alert">
            <p class="font-bold">Success</p>
            <p>' . htmlspecialchars($_SESSION['success_message']) . '</p>
          </div>';
    unset($_SESSION['success_message']);
}

// 11. Include the Page Content
// This is where pages/dashboard.php, pages/submit_post.php, etc., are loaded.
include $page_file;

// 12. Load Footer Template
// The footer contains all closing HTML tags
require_once 'templates/footer.php';
?>

