<?php
// FILE: includes/functions2.php
// This file contains all "action" handlers that modify the database.

// === USER & AUTH HANDLERS ===

/**
 * Handles user login.
 * @return string An error message on failure. Redirects on success.
 */
function handleLogin() {
    validateCSRF();
    $db = getDB();
    $email = $_POST['email'];
    $password = $_POST['password'];

    try {
        $stmt = $db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            
            // --- NEW PENDING LOGIC ---
            // If user is pending, log them in but force them to their profile.
            $isPending = ($user['status'] === 'pending' || $user['status'] === 'pending_admin');

            session_regenerate_id(true);
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_role'] = $user['role'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['name'];
            $_SESSION['user_status'] = $user['status']; // Store status in session
            
            if ($isPending) {
                // Force pending users to their profile page
                header("Location: index.php?page=my_profile");
                exit;
            }
            // --- END NEW PENDING LOGIC ---

            if ($user['status'] === 'approved') {
                // Redirect to the intended page or dashboard
                $redirect_url = $_SESSION['redirect_url'] ?? 'index.php?page=dashboard';
                unset($_SESSION['redirect_url']);
                
                header("Location: $redirect_url");
                exit;
            }
            
            return "Your account is not active. Please contact an admin.";

        } else {
            return "Invalid email or password.";
        }
    } catch (PDOException $e) {
        error_log("Login PDOException: " . $e->getMessage());
        return "A database error occurred. Please try again later.";
    }
}

/**
 * Handles new user registration (multi-step).
 * Sets session messages.
 */
function handleRegister() {
    validateCSRF();
    $db = getDB();

    try {
        // === Step 1: Account Data ===
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'blogger';
        $supervisor_id = ($role === 'blogger') ? ($_POST['supervisorId'] ?? null) : null;
        
        // === Step 2: Profile Data ===
        $phone_number = trim($_POST['phone_number'] ?? '');
        $payment_number = trim($_POST['payment_number'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $bio = trim($_POST['bio'] ?? '');
        $expertise = trim($_POST['expertise'] ?? '');

        // === Step 3: Social Data ===
        $socials = $_POST['socials'] ?? [];
        
        // --- Validation ---
        if (empty($name) || empty($email) || empty($password) || empty($role)) {
            $_SESSION['error_message'] = "Step 1 is incomplete. Please fill all required fields.";
            return;
        }
        if (empty($phone_number) || empty($payment_number) || empty($location)) {
            $_SESSION['error_message'] = "Step 2 is incomplete. Please fill all required fields.";
            return;
        }
        if ($role === 'blogger') {
            $stmt = $db->query("SELECT COUNT(*) FROM users WHERE role IN ('supervisor', 'admin') AND status = 'approved'");
            if ($stmt->fetchColumn() > 0 && empty($supervisor_id)) {
                // Only require supervisor if supervisors actually exist and one wasn't chosen
                $_SESSION['error_message'] = "As a blogger, you must select a supervisor.";
                return;
            }
        }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['error_message'] = "Invalid email format.";
            return;
        }
        if (strlen($password) < 8) {
            $_SESSION['error_message'] = "Password must be at least 8 characters long.";
            return;
        }

        // Check if user exists
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetchColumn() > 0) {
            $_SESSION['error_message'] = "An account with this email already exists.";
            return;
        }
        
        // --- Set Status based on Role ---
        $status = 'pending'; // Default for bloggers
        if ($role === 'supervisor') {
            $status = 'pending_admin'; // Supervisors need admin approval
        }

        $db->beginTransaction();

        // 1. Insert into 'users' table
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $db->prepare("
            INSERT INTO users 
                (name, email, password, role, status, supervisor_id, phone_number, payment_number, location, bio, expertise)
            VALUES 
                (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $name, $email, $hashedPassword, $role, $status, $supervisor_id,
            $phone_number, $payment_number, $location, $bio, $expertise
        ]);
        
        $user_id = $db->lastInsertId();

        // 2. Insert into 'blogger_socials' table (if blogger)
        if ($role === 'blogger' && !empty($socials)) {
            $socialStmt = $db->prepare("
                INSERT INTO blogger_socials
                    (user_id, platform, url, followers, is_verified)
                VALUES
                    (?, ?, ?, ?, ?)
            ");
            
            foreach ($socials as $social) {
                if (!empty($social['url'])) {
                    $platform = $social['platform'] ?? 'other';
                    $url = $social['url'];
                    $followers = (int)($social['followers'] ?? 0);
                    $is_verified = (isset($social['is_verified']) && $social['is_verified'] == '1') ? 1 : 0;
                    
                    $socialStmt->execute([$user_id, $platform, $url, $followers, $is_verified]);
                }
            }
        }
        
        $db->commit();
        
        if ($status === 'pending_admin') {
            $_SESSION['success_message'] = "Registration successful! Your Supervisor account is pending Admin approval. You can now log in to update your profile.";
        } else {
            $_SESSION['success_message'] = "Registration successful! Your account is pending approval by your supervisor. You can now log in to update your profile.";
        }
        
    } catch (PDOException $e) {
        $db->rollBack();
        error_log("Register PDOException: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred during registration. " . $e->getMessage();
    }
}


/**
 * Approves a new user (blogger or supervisor).
 * Sets session messages.
 */
function handleApproveUser() {
    checkSupervisor(); // At least a supervisor
    $db = getDB();
    $user_id = $_POST['user_id'];
    $current_user_id = $_SESSION['user_id'];
    $current_user_role = $_SESSION['user_role'];

    try {
        // Get the user we are trying to approve
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userToApprove = $stmt->fetch();

        if (!$userToApprove) {
            $_SESSION['error_message'] = "User not found.";
            return;
        }

        // --- Authorization Logic ---
        if ($current_user_role === 'admin') {
            // Admin can approve ANY pending user ('pending' or 'pending_admin')
            if ($userToApprove['status'] === 'pending' || $userToApprove['status'] === 'pending_admin') {
                 $stmt = $db->prepare("UPDATE users SET status = 'approved' WHERE id = ?");
                 $stmt->execute([$user_id]);
                 $_SESSION['success_message'] = "User approved successfully.";
            } else {
                 $_SESSION['error_message'] = "This user is not pending approval.";
            }
        } elseif ($current_user_role === 'supervisor') {
            // Supervisor can ONLY approve 'pending' bloggers assigned to them
            if ($userToApprove['role'] === 'blogger' && 
                $userToApprove['status'] === 'pending' && 
                $userToApprove['supervisor_id'] == $current_user_id) 
            {
                $stmt = $db->prepare("UPDATE users SET status = 'approved' WHERE id = ?");
                $stmt->execute([$user_id]);
                $_SESSION['success_message'] = "Blogger approved successfully.";
            } else {
                // Any other case is unauthorized
                $_SESSION['error_message'] = "You are not authorized to approve this user.";
            }
        }
        
    } catch (PDOException $e) {
        error_log("Approve User Error: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred.";
    }
}


/**
 * Rejects a new user.
 * Sets session messages.
 */
function handleRejectUser() {
    checkSupervisor();
    $db = getDB();
    $user_id = $_POST['user_id'];
    $current_user_id = $_SESSION['user_id'];
    $current_user_role = $_SESSION['user_role'];

    try {
        // Get the user we are trying to reject
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userToReject = $stmt->fetch();

        if (!$userToReject) {
            $_SESSION['error_message'] = "User not found.";
            return;
        }
        
        $allowedStatuses = ['pending', 'pending_admin'];
        if (!in_array($userToReject['status'], $allowedStatuses)) {
             $_SESSION['error_message'] = "This user is not pending approval.";
             return;
        }

        // --- Authorization Logic ---
        if ($current_user_role === 'admin') {
            // Admin can reject ANY pending user
            $stmt = $db->prepare("DELETE FROM users WHERE id = ? AND status IN ('pending', 'pending_admin')");
            $stmt->execute([$user_id]);
        } elseif ($current_user_role === 'supervisor') {
            // Supervisor can ONLY reject 'pending' bloggers assigned to them
            if ($userToReject['role'] === 'blogger' && $userToReject['supervisor_id'] == $current_user_id) {
                $stmt = $db->prepare("DELETE FROM users WHERE id = ? AND status = 'pending'");
                $stmt->execute([$user_id]);
            } else {
                $_SESSION['error_message'] = "You are not authorized to reject this user.";
                return;
            }
        }
        
        if ($stmt->rowCount() > 0) {
            $_SESSION['success_message'] = "User rejected and removed.";
        } else {
            $_SESSION['error_message'] = "Could not reject user.";
        }
    } catch (PDOException $e) {
        error_log("Reject User Error: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred. The user may have associated data.";
    }
}

/**
 * Changes a user's role. Admin only.
 * Sets session messages.
 */
function handleSetRole() {
    checkAdmin();
    $db = getDB();
    $user_id = $_POST['user_id'];
    $newRole = $_POST['new_role'];

    if (!in_array($newRole, ['blogger', 'supervisor', 'admin'])) {
        $_SESSION['error_message'] = "Invalid role selected.";
        return;
    }
    
    // Prevent admin from demoting themselves (if they are the only admin)
    if ($user_id == $_SESSION['user_id'] && $newRole !== 'admin') {
        $stmt = $db->query("SELECT COUNT(*) FROM users WHERE role = 'admin'");
        if ($stmt->fetchColumn() <= 1) {
            $_SESSION['error_message'] = "You cannot demote the last admin account.";
            return;
        }
    }

    try {
        $stmt = $db->prepare("UPDATE users SET role = ? WHERE id = ?");
        $stmt->execute([$newRole, $user_id]);
        $_SESSION['success_message'] = "User role updated successfully.";
    } catch (PDOException $e) {
        error_log("Set Role Error: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred.";
    }
}


// === POST & EARNINGS HANDLERS ===

/**
 * Handles submission of a new post.
 * Sets session messages.
 */
function handleSubmitPost() {
    // Role check
    if ($_SESSION['user_role'] !== 'blogger') {
        $_SESSION['error_message'] = "Only bloggers can submit posts.";
        return;
    }
    
    $db = getDB();
    $blogger_id = $_SESSION['user_id'];
    $url = $_POST['url'] ?? '';
    
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        $_SESSION['error_message'] = "A valid Post URL is required.";
        return;
    }

    try {
        // We no longer use screenshot_url
        $stmt = $db->prepare("INSERT INTO posts (blogger_id, url, status) VALUES (?, ?, 'pending')");
        $stmt->execute([$blogger_id, $url]);
        
        $_SESSION['success_message'] = "Thank you for submitting your post, your supervisor will review. Keep going.";
    } catch (PDOException $e) {
        error_log("handleSubmitPost Error: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred while submitting your post. Please try again.";
    }
}

/**
 * First-step approval: Moves post to 'performance_review'.
 * Sets session messages.
 */
function handleApprovePost() {
    checkSupervisor();
    $db = getDB();
    
    $post_id = $_POST['post_id'];
    $supervisor_id = $_SESSION['user_id'];

    try {
        // 1. Get post and blogger info
        $stmt = $db->prepare("
            SELECT p.status, u.supervisor_id 
            FROM posts p 
            JOIN users u ON p.blogger_id = u.id 
            WHERE p.id = ?
        ");
        $stmt->execute([$post_id]);
        $post = $stmt->fetch();

        if (!$post) {
            $_SESSION['error_message'] = "Post not found.";
            return;
        }
        if ($post['status'] !== 'pending') {
            $_SESSION['error_message'] = "This post has already been reviewed.";
            return;
        }

        // 2. Authorization check
        if ($_SESSION['user_role'] !== 'admin' && $supervisor_id != $post['supervisor_id']) {
            $_SESSION['error_message'] = "You are not authorized to approve this post.";
            return;
        }
        
        // 3. Update post status to 'performance_review' and set approval time
        $stmt = $db->prepare("
            UPDATE posts 
            SET status = 'performance_review', approved_at = CURRENT_TIMESTAMP, approved_by = ? 
            WHERE id = ?
        ");
        $stmt->execute([$supervisor_id, $post_id]);
        
        $_SESSION['success_message'] = "Post approved for performance review. It will be live on the feed and ready for final review in 24 hours.";
        
    } catch (Exception $e) {
        error_log("Approve Post Error: " . $e->getMessage());
        $_SESSION['error_message'] = "Error approving post: " . $e->getMessage();
    }
}

/**
 * Rejects a pending post.
 * Sets session messages.
 */
function handleRejectPost() {
    checkSupervisor();
    $db = getDB();
    $post_id = $_POST['post_id'];
    $supervisor_id = $_SESSION['user_id'];

    try {
        // 1. Get post and blogger info
        $stmt = $db->prepare("
            SELECT p.status, u.supervisor_id 
            FROM posts p 
            JOIN users u ON p.blogger_id = u.id 
            WHERE p.id = ?
        ");
        $stmt->execute([$post_id]);
        $post = $stmt->fetch();

        if (!$post) {
            $_SESSION['error_message'] = "Post not found.";
            return;
        }
        if ($post['status'] !== 'pending') {
            $_SESSION['error_message'] = "This post has already been reviewed.";
            return;
        }

        // 2. Authorization check
        if ($_SESSION['user_role'] !== 'admin' && $supervisor_id != $post['supervisor_id']) {
            $_SESSION['error_message'] = "You are not authorized to reject this post.";
            return;
        }

        // 3. Update post status
        $stmt = $db->prepare("UPDATE posts SET status = 'rejected' WHERE id = ?");
        $stmt->execute([$post_id]);
        
        $_SESSION['success_message'] = "Post rejected successfully.";
        
    } catch (PDOException $e) {
        error_log("Reject Post Error: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred.";
    }
}


/**
 * Final performance review: Calculates earnings based on new logic.
 * Sets session messages.
 */
function handlePerformanceReview() {
    checkSupervisor();
    $db = getDB();

    // Data from form
    $post_id = $_POST['post_id'];
    $reach = (int)($_POST['reach'] ?? 0);
    $likes = (int)($_POST['likes'] ?? 0);
    $comments = (int)($_POST['comments'] ?? 0);
    $shares = (int)($_POST['shares'] ?? 0);
    
    // Auth
    $supervisor_id = $_SESSION['user_id'];

    // --- Earning Logic Constants ---
    $REACH_THRESHOLD = 1000;
    $LIKE_RATE = 1;
    $LIKE_CAP = 300;
    $COMMENT_RATE = 2;
    $COMMENT_CAP = 300;
    $SHARE_RATE = 5;
    $SHARE_CAP = 300;
    $COMMISSION_RATE = 0.10; // 10%

    try {
        // 1. Get post and blogger info
        $stmt = $db->prepare("
            SELECT p.blogger_id, p.status, u.supervisor_id 
            FROM posts p 
            JOIN users u ON p.blogger_id = u.id 
            WHERE p.id = ?
        ");
        $stmt->execute([$post_id]);
        $post = $stmt->fetch();

        if (!$post) {
            $_SESSION['error_message'] = "Post not found.";
            return;
        }
        if ($post['status'] !== 'performance_review') {
            $_SESSION['error_message'] = "This post is not awaiting performance review.";
            return;
        }

        // 2. Authorization check
        $blogger_supervisor_id = $post['supervisor_id'];
        if ($_SESSION['user_role'] !== 'admin' && $supervisor_id != $blogger_supervisor_id) {
            $_SESSION['error_message'] = "You are not authorized to review this post.";
            return;
        }
        
        // 3. Check 24-hour window (already done by the query in functions.php, but good to double check)
        // ... (Skipped for brevity, as getPerformanceReviewPosts should handle this)

        $db->beginTransaction();
        
        // 4. Update post status to 'completed'
        $stmt = $db->prepare("UPDATE posts SET status = 'completed' WHERE id = ?");
        $stmt->execute([$post_id]);
        
        // 5. Insert metrics
        $stmt = $db->prepare("INSERT INTO post_metrics (post_id, reach, likes, comments, shares) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$post_id, $reach, $likes, $comments, $shares]);
        
        // 6. --- Calculate Earnings ---
        $bloggerEarnings = 0;
        if ($reach >= $REACH_THRESHOLD) {
            // Calculate earnings for likes, comments, and shares, applying caps
            $likeEarnings = min($likes * $LIKE_RATE, $LIKE_CAP);
            $commentEarnings = min($comments * $COMMENT_RATE, $COMMENT_CAP);
            $shareEarnings = min($shares * $SHARE_RATE, $SHARE_CAP);
            
            $bloggerEarnings = $likeEarnings + $commentEarnings + $shareEarnings;
        }
        
        // 7. Calculate commission
        $supervisorCommission = $bloggerEarnings * $COMMISSION_RATE;
        
        $earningsStmt = $db->prepare("
            INSERT INTO earnings 
                (user_id, post_id, type, amount, from_blogger_id, review_type) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $blogger_id = $post['blogger_id'];
        
        // 8. Insert blogger earnings record
        if ($bloggerEarnings > 0) {
            $earningsStmt->execute([
                $blogger_id, $post_id, 'post_earning', $bloggerEarnings, NULL, 'performance'
            ]);
        }
        
        // 9. Insert supervisor commission record
        if ($supervisorCommission > 0) {
            $earningsStmt->execute([
                $blogger_supervisor_id, $post_id, 'commission', $supervisorCommission, $blogger_id, 'performance'
            ]);
        }
        
        $db->commit();
        
        if ($bloggerEarnings > 0) {
             $_SESSION['success_message'] = "Performance review complete. Blogger earned KES " . number_format($bloggerEarnings, 2) . ".";
        } else if ($reach < $REACH_THRESHOLD) {
             $_SESSION['success_message'] = "Review complete. Post did not meet the 1000 reach threshold, no earnings calculated.";
        } else {
             $_SESSION['success_message'] = "Performance review complete. No earnings calculated.";
        }
        
    } catch (Exception $e) {
        $db->rollBack();
        error_log("Performance Review Error: " . $e->getMessage());
        $_SESSION['error_message'] = "Error during performance review: " . $e->getMessage();
    }
}


// === COMMUNITY (PHASE 2) HANDLERS ===
// These are no longer used since we switched to external linking.

/**
 * Adds a comment to a post. (DEPRECATED)
 */
function handleAddComment() {
    // This logic is no longer used.
    $_SESSION['error_message'] = "This feature is no longer active.";
    return;
}

/**
 * Likes or unlikes a post. (DEPRECATED)
 */
function handleLikePost() {
    // This logic is no longer used.
    $_SESSION['error_message'] = "This feature is no longer active.";
    return;
}


// === NEW PROFILE PAGE HANDLER ===

/**
 * Handles updates to a user's own profile.
 * Sets session messages.
 */
function handleUpdateProfile() {
    validateCSRF();
    $db = getDB();
    $user_id = $_SESSION['user_id'];

    try {
        // === Profile Data ===
        $name = trim($_POST['name'] ?? '');
        $phone_number = trim($_POST['phone_number'] ?? '');
        $payment_number = trim($_POST['payment_number'] ?? '');
        $location = trim($_POST['location'] ?? '');
        $bio = trim($_POST['bio'] ?? '');
        $expertise = trim($_POST['expertise'] ?? '');

        // === Social Data ===
        $socials = $_POST['socials'] ?? [];
        
        // --- Validation ---
        if (empty($name) || empty($phone_number) || empty($payment_number) || empty($location)) {
            $_SESSION['error_message'] = "Please fill all required profile fields.";
            return;
        }
        
        $db->beginTransaction();

        // 1. Update 'users' table
        $stmt = $db->prepare("
            UPDATE users 
            SET name = ?, phone_number = ?, payment_number = ?, location = ?, bio = ?, expertise = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $name, $phone_number, $payment_number, $location, $bio, $expertise,
            $user_id
        ]);
        
        // Update session name
        $_SESSION['user_name'] = $name;
        
        // 2. Update 'blogger_socials' table
        if ($_SESSION['user_role'] === 'blogger') {
            
            // For simplicity, we'll delete all old socials and re-insert
            // A more complex app might do row-by-row updates
            $deleteStmt = $db->prepare("DELETE FROM blogger_socials WHERE user_id = ?");
            $deleteStmt->execute([$user_id]);
            
            $socialStmt = $db->prepare("
                INSERT INTO blogger_socials
                    (user_id, platform, url, followers, is_verified)
                VALUES
                    (?, ?, ?, ?, ?)
            ");
            
            if (!empty($socials)) {
                foreach ($socials as $social) {
                    if (!empty($social['url'])) {
                        $platform = $social['platform'] ?? 'other';
                        $url = $social['url'];
                        $followers = (int)($social['followers'] ?? 0);
                        // Handle checkbox value
                        $is_verified = (isset($social['is_verified']) && $social['is_verified'] == '1') ? 1 : 0;
                        
                        $socialStmt->execute([$user_id, $platform, $url, $followers, $is_verified]);
                    }
                }
            }
        }
        
        $db->commit();
        
        $_SESSION['success_message'] = "Your profile has been updated successfully.";
        
    } catch (PDOException $e) {
        $db->rollBack();
        error_log("Update Profile PDOException: " . $e->getMessage());
        $_SESSION['error_message'] = "A database error occurred while updating your profile.";
    }
}
?>

