<?php
// FILE: includes/functions.php
// This file contains all core logic, database queries, and helper functions.

// We need the db connection, but it's included *before* this file in index.php and login.php
// No need to include db.php here, as it would cause a redeclare error.


// === SESSION & SECURITY FUNCTIONS ===

/**
 * Starts a secure PHP session.
 */
function secure_session_start() {
    $session_name = 'genowa_session_id'; // Set a custom session name
    $secure = false; // Set to true if using HTTPS
    $httponly = true; // Prevents JavaScript from accessing the session cookie

    // Get current cookie params
    $cookieParams = session_get_cookie_params();
    session_set_cookie_params(
        $cookieParams["lifetime"],
        $cookieParams["path"],
        $cookieParams["domain"],
        $secure,
        $httponly
    );
    
    session_name($session_name);
    
    // Start the session
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    // Regenerate session ID periodically to prevent session fixation
    if (!isset($_SESSION['last_regen'])) {
        $_SESSION['last_regen'] = time();
    } elseif (time() - $_SESSION['last_regen'] > 1800) { // 30 minutes
        session_regenerate_id(true);
        $_SESSION['last_regen'] = time();
    }
}

/**
 * Generates or retrieves a CSRF token.
 * @return string
 */
function getCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Validates the CSRF token from a POST request.
 * Dies if validation fails.
 */
function validateCSRF() {
    if (empty($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        // Log the attempt
        error_log("CSRF Token Validation Failed. IP: " . $_SERVER['REMOTE_ADDR']);
        // Set a user-friendly error message and redirect
        $_SESSION['error_message'] = 'Security token invalid. Please try again.';
        // Redirect back to the page they came from
        header("Location: " . $_SERVER['HTTP_REFERER'] ?? 'index.php');
        exit;
    }
}

/**
 * Checks if a user is authenticated. If not, redirects to login.
 * @return array The user's role and status if authenticated.
 */
function checkAuth() {
    if (!isset($_SESSION['user_id'])) {
        // Store the intended page to redirect after login
        $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
        header("Location: login.php");
        exit;
    }
    return [
        'role' => $_SESSION['user_role'],
        'status' => $_SESSION['user_status']
    ];
}

/**
 * Checks if user is at least a Supervisor and approved.
 * Redirects to dashboard if not.
 */
function checkSupervisor() {
    $auth = checkAuth();
    if ($auth['status'] !== 'approved' || !in_array($auth['role'], ['supervisor', 'admin'])) {
        $_SESSION['error_message'] = 'You do not have permission to access that page.';
        header("Location: index.php?page=my_profile"); // Redirect to profile
        exit;
    }
}

/**
 * Checks if user is an Admin and approved.
 * Redirects to dashboard if not.
 */
function checkAdmin() {
    $auth = checkAuth();
    if ($auth['status'] !== 'approved' || $auth['role'] !== 'admin') {
        $_SESSION['error_message'] = 'You do not have permission to access that page.';
        header("Location: index.php?page=my_profile"); // Redirect to profile
        exit;
    }
}


// === DATA FETCHER FUNCTIONS (for pages) ===

/**
 * Gets a list of all supervisors for the registration form.
 * @return array
 */
function getSupervisors() {
    $db = getDB();
    // Only show APPROVED supervisors whose names are NOT empty
    $stmt = $db->query("
        SELECT id, name 
        FROM users 
        WHERE role IN ('supervisor', 'admin') 
        AND status = 'approved' 
        AND name IS NOT NULL 
        AND name != ''
        ORDER BY name
    ");
    return $stmt->fetchAll();
}

/**
 * Gets all user profile data.
 * @param int $user_id
 * @return array|false
 */
function getUserProfile($user_id) {
    $db = getDB();
    $stmt = $db->prepare("SELECT id, name, email, location, phone_number, payment_number, bio, expertise, status FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    return $stmt->fetch();
}

/**
 * Gets all social media links for a user.
 * @param int $user_id
 * @return array
 */
function getUserSocials($user_id) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM blogger_socials WHERE user_id = ? ORDER BY id ASC");
    $stmt->execute([$user_id]);
    return $stmt->fetchAll();
}

/**
 * Gets a list of all users for the admin manage users page.
 * @return array
 */
function getAllUsers() {
    $db = getDB();
    $stmt = $db->query("
        SELECT u.id, u.name, u.email, u.role, u.status, s.name as supervisor_name
        FROM users u
        LEFT JOIN users s ON u.supervisor_id = s.id
        ORDER BY u.name
    ");
    return $stmt->fetchAll();
}

/**
 * Gets a list of pending bloggers and supervisors for approval.
 * @param int $supervisor_id
 * @param string $role
 * @return array
 */
function getPendingTeam($supervisor_id, $role) {
    $db = getDB();
    if ($role === 'admin') {
        // Admins see all pending bloggers AND pending supervisors
        $stmt = $db->query("
            SELECT u.id, u.name, u.email, u.role, s.name as supervisor_name, u.created_at
            FROM users u
            LEFT JOIN users s ON u.supervisor_id = s.id
            WHERE u.status = 'pending' OR u.status = 'pending_admin'
            ORDER BY u.created_at ASC
        ");
    } else {
        // Supervisors see only their pending bloggers
        $stmt = $db->prepare("
            SELECT id, name, email, role, created_at 
            FROM users 
            WHERE status = 'pending' AND role = 'blogger' AND supervisor_id = ?
            ORDER BY created_at ASC
        ");
        $stmt->execute([$supervisor_id]);
    }
    return $stmt->fetchAll();
}

/**
 * Gets a list of pending posts for a supervisor.
 * @param int $supervisor_id
 * @param string $role
 * @return array
 */
function getPendingPostsForSupervisor($supervisor_id, $role) {
    $db = getDB();
    if ($role === 'admin') {
        // Admins see all pending posts
        $sql = "SELECT p.id, p.url, u.name as blogger_name, u.email as blogger_email
                FROM posts p
                JOIN users u ON p.blogger_id = u.id
                WHERE p.status = 'pending'
                ORDER BY p.submitted_at ASC";
        $stmt = $db->query($sql);
    } else {
        // Supervisors see only their team's pending posts
        $sql = "SELECT p.id, p.url, u.name as blogger_name, u.email as blogger_email
                FROM posts p
                JOIN users u ON p.blogger_id = u.id
                WHERE p.status = 'pending' AND u.supervisor_id = ?
                ORDER BY p.submitted_at ASC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$supervisor_id]);
    }
    return $stmt->fetchAll();
}

/**
 * Gets a list of posts for performance review (approved > 24 hours ago).
 * @param int $supervisor_id
 * @param string $role
 * @return array
 */
function getPerformanceReviewPosts($supervisor_id, $role) {
    $db = getDB();
    // Get posts that are 'performance_review' AND approved_at is older than 24 hours
    $sql = "
        SELECT p.id, p.url, u.name as blogger_name, u.email as blogger_email, p.approved_at
        FROM posts p
        JOIN users u ON p.blogger_id = u.id
        WHERE p.status = 'performance_review'
        AND p.approved_at <= datetime('now', '-24 hours')
    ";

    if ($role === 'admin') {
        // Admin sees all
        $sql .= " ORDER BY p.approved_at ASC";
        $stmt = $db->query($sql);
    } else {
        // Supervisor sees only their team
        $sql .= " AND u.supervisor_id = ? ORDER BY p.approved_at ASC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$supervisor_id]);
    }
    return $stmt->fetchAll();
}


// === DASHBOARD DATA FUNCTIONS ===

/**
 * Gets stats for the blogger dashboard.
 * @param int $blogger_id
 * @return array
 */
function getBloggerStats($blogger_id) {
    $db = getDB();
    $stats = ['earnings' => 0, 'reach' => 0, 'posts' => 0, 'points' => 0];

    // Total Earnings (both initial and performance)
    $stmt = $db->prepare("SELECT SUM(amount) FROM earnings WHERE user_id = ? AND type = 'post_earning'");
    $stmt->execute([$blogger_id]);
    $stats['earnings'] = $stmt->fetchColumn() ?? 0;

    // Total Reach & Completed Posts
    $stmt = $db->prepare("
        SELECT SUM(m.reach) as total_reach, COUNT(p.id) as total_posts
        FROM posts p
        JOIN post_metrics m ON p.id = m.post_id
        WHERE p.blogger_id = ? AND p.status = 'completed'
    ");
    $stmt->execute([$blogger_id]);
    $postStats = $stmt->fetch();
    $stats['reach'] = $postStats['total_reach'] ?? 0;
    $stats['posts'] = $postStats['total_posts'] ?? 0;
    
    // Total Interaction Points
    $stmt = $db->prepare("SELECT SUM(points) FROM points WHERE user_id = ?");
    $stmt->execute([$blogger_id]);
    $stats['points'] = $stmt->fetchColumn() ?? 0;

    return $stats;
}

/**
 * Gets stats for the supervisor dashboard.
 * @param int $supervisor_id
 * @return array
 */
function getSupervisorStats($supervisor_id) {
    $db = getDB();
    $stats = [
        'earnings' => 0, // My personal post earnings
        'commissions' => 0,
        'teamReach' => 0,
        'teamPosts' => 0
    ];
    
    // My personal post earnings
    $stmt = $db->prepare("SELECT SUM(amount) FROM earnings WHERE user_id = ? AND type = 'post_earning'");
    $stmt->execute([$supervisor_id]);
    $stats['earnings'] = $stmt->fetchColumn() ?? 0;
    
    // My commission earnings
    $stmt = $db->prepare("SELECT SUM(amount) FROM earnings WHERE user_id = ? AND type = 'commission'");
    $stmt->execute([$supervisor_id]);
    $stats['commissions'] = $stmt->fetchColumn() ?? 0;

    // Get team member IDs
    $teamStmt = $db->prepare("SELECT id FROM users WHERE supervisor_id = ? AND status = 'approved'");
    $teamStmt->execute([$supervisor_id]);
    $teamIds = $teamStmt->fetchAll(PDO::FETCH_COLUMN);

    if (!empty($teamIds)) {
        $placeholders = implode(',', array_fill(0, count($teamIds), '?'));
        
        $stmt = $db->prepare("
            SELECT SUM(m.reach) as total_reach, COUNT(p.id) as total_posts
            FROM posts p
            JOIN post_metrics m ON p.id = m.post_id
            WHERE p.blogger_id IN ($placeholders) AND p.status = 'completed'
        ");
        $stmt->execute($teamIds);
        $teamStats = $stmt->fetch();
        
        $stats['teamReach'] = $teamStats['total_reach'] ?? 0;
        $stats['teamPosts'] = $teamStats['total_posts'] ?? 0;
    }
    
    return $stats;
}

/**
 * Gets stats for the admin dashboard.
 * @return array
 */
function getAdminStats() {
    $db = getDB();
    $stats = [
        'totalEarnings' => 0,
        'totalReach' => 0,
        'totalPosts' => 0,
        'totalUsers' => 0
    ];
    
    $stats['totalEarnings'] = $db->query("SELECT SUM(amount) FROM earnings")->fetchColumn() ?? 0;
    $stats['totalReach'] = $db->query("SELECT SUM(reach) FROM post_metrics")->fetchColumn() ?? 0;
    $stats['totalPosts'] = $db->query("SELECT COUNT(*) FROM posts WHERE status = 'completed'")->fetchColumn() ?? 0;
    $stats['totalUsers'] = $db->query("SELECT COUNT(*) FROM users WHERE status = 'approved'")->fetchColumn() ?? 0;
    
    return $stats;
}

/**
 * Gets recent posts for the blogger dashboard.
 * @param int $blogger_id
 * @return array
 */
function getBloggerRecentPosts($blogger_id) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT url, submitted_at, status 
        FROM posts 
        WHERE blogger_id = ? 
        ORDER BY submitted_at DESC 
        LIMIT 5
    ");
    $stmt->execute([$blogger_id]);
    return $stmt->fetchAll();
}

/**
 * Gets detailed stats for each team member for the supervisor dashboard.
 * @param int $supervisor_id
 * @return array
 */
function getSupervisorTeamDetails($supervisor_id) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT 
            u.id, 
            u.name, 
            u.email, 
            u.status,
            COALESCE(SUM(e.amount), 0) as earnings,
            COALESCE(SUM(pm.reach), 0) as reach,
            COUNT(DISTINCT p.id) as posts
        FROM users u
        LEFT JOIN posts p ON u.id = p.blogger_id AND p.status = 'completed'
        LEFT JOIN post_metrics pm ON p.id = pm.post_id
        LEFT JOIN earnings e ON u.id = e.user_id AND e.type = 'post_earning'
        WHERE u.supervisor_id = ? AND u.role = 'blogger'
        GROUP BY u.id, u.name, u.email, u.status
        ORDER BY earnings DESC, u.email ASC
    ");
    $stmt->execute([$supervisor_id]);
    return $stmt->fetchAll();
}

/**
 * Gets an overview of all supervisors for the admin dashboard.
 * @return array
 */
function getAdminSupervisorOverview() {
    $db = getDB();
    $stmt = $db->query("
        SELECT 
            s.id, 
            s.name, 
            s.email,
            COUNT(DISTINCT t.id) as team_size,
            COALESCE(SUM(e.amount), 0) as commissions,
            COALESCE(SUM(pm.reach), 0) as team_reach
        FROM users s
        LEFT JOIN users t ON s.id = t.supervisor_id AND t.role = 'blogger' AND t.status = 'approved'
        LEFT JOIN earnings e ON s.id = e.user_id AND e.type = 'commission'
        LEFT JOIN posts p ON t.id = p.blogger_id AND p.status = 'completed'
        LEFT JOIN post_metrics pm ON p.id = pm.post_id
        WHERE s.role IN ('supervisor', 'admin')
        GROUP BY s.id, s.name, s.email
        ORDER BY commissions DESC, s.email ASC
    ");
    return $stmt->fetchAll();
}


// === COMMUNITY (PHASE 2) DATA FUNCTIONS ===

/**
 * Gets the main community feed of approved posts.
 * @param int $current_user_id
 * @return array
 */
function getCommunityFeed($current_user_id) {
    $db = getDB();
    // Show posts that are either 'performance_review' or 'completed'
    $stmt = $db->prepare("
        SELECT 
            p.id, 
            p.url, 
            p.submitted_at,
            u.name as blogger_name,
            COALESCE(pm.reach, 0) as reach,
            COALESCE(pm.likes, 0) as likes,
            COALESCE(pm.comments, 0) as comments,
            COALESCE(pm.shares, 0) as shares
        FROM posts p
        JOIN users u ON p.blogger_id = u.id
        LEFT JOIN post_metrics pm ON p.id = pm.post_id
        WHERE p.status = 'performance_review' OR p.status = 'completed'
        ORDER BY p.submitted_at DESC
    ");
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Gets all data for a single post.
 * @param int $post_id
 * @param int $current_user_id
 * @return array|false
 */
function getSinglePost($post_id, $current_user_id) {
    $db = getDB();
    // This function is no longer used for interaction, just for viewing.
    // We can simplify it or remove it. For now, let's assume it's
    // just for viewing the post details.
    $stmt = $db->prepare("
        SELECT 
            p.id, 
            p.url, 
            p.submitted_at,
            u.name as blogger_name,
            COALESCE(pm.reach, 0) as reach,
            COALESCE(pm.likes, 0) as likes,
            COALESCE(pm.comments, 0) as comments,
            COALESCE(pm.shares, 0) as shares
        FROM posts p
        JOIN users u ON p.blogger_id = u.id
        LEFT JOIN post_metrics pm ON p.id = pm.post_id
        WHERE p.id = ? AND (p.status = 'performance_review' OR p.status = 'completed')
    ");
    $stmt->execute([$post_id]);
    $post = $stmt->fetch();
    
    // We no longer need internal comments/likes, so we'll remove those queries.
    return $post;
}

/**
 * Gets all comments for a single post. (No longer used internally)
 * @param int $post_id
 * @return array
 */
function getCommentsForPost($post_id) {
    // This logic is now external (on Facebook, etc.)
    return [];
}

?>

