<?php
// FILE: api/dashboard_data_api.php
// --- CORRECTED & STABILIZED ---

require_once '../db_connect.php';

header('Content-Type: application/json');

// --- Filter Setup ---
$constituency_id = $_GET['constituency_id'] ?? null;
$ward_id = $_GET['ward_id'] ?? null;
$pillar_id = $_GET['pillar_id'] ?? null;
$report_title = $_GET['report_title'] ?? 'The Entire County'; // Get title from frontend

$base_sql = "
    SELECT 
        p.status, p.expected_completion_date, p.actual_completion_date,
        mp.title as pillar_title, c.name as constituency_name, w.name as ward_name,
        p.budget_allocated, p.budget_spent
    FROM projects p
    LEFT JOIN wards w ON p.ward_id = w.id
    LEFT JOIN constituencies c ON w.constituency_id = c.id
    LEFT JOIN manifesto_pillars mp ON p.manifesto_pillar_id = mp.id
";

$where_clauses = ["p.is_approved = 1"];
$params = [];

if (!empty($constituency_id)) {
    $where_clauses[] = "c.id = :constituency_id";
    $params[':constituency_id'] = $constituency_id;
}
if (!empty($ward_id)) {
    $where_clauses[] = "w.id = :ward_id";
    $params[':ward_id'] = $ward_id;
}
if (!empty($pillar_id)) {
    $where_clauses[] = "p.manifesto_pillar_id = :pillar_id";
    $params[':pillar_id'] = $pillar_id;
}

$sql_where = " WHERE " . implode(" AND ", $where_clauses);
$final_sql = $base_sql . $sql_where;

try {
    $stmt = $pdo->prepare($final_sql);
    $stmt->execute($params);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $issue_stmt = $pdo->query("SELECT status, COUNT(id) as count FROM issues_and_contacts GROUP BY status");
    $issue_status_counts = $issue_stmt->fetchAll(PDO::FETCH_KEY_PAIR);

    $dashboard_data = [
        'report_title' => $report_title,
        'kpis' => ['total_projects' => 0, 'total_investment' => 0, 'avg_delay' => 0],
        'status' => [], 'sector' => [], 'location' => [],
        'financials' => ['allocated' => 0, 'spent' => 0],
        'timeliness' => ['On Time' => 0, 'Late' => 0],
        'issueStatus' => $issue_status_counts, 'budgetUtilization' => []
    ];

    $total_delay = 0;
    $late_projects_count = 0;

    foreach ($projects as $project) {
        $dashboard_data['kpis']['total_projects']++;
        $investment = (float)($project['budget_allocated'] ?? 0);
        $dashboard_data['kpis']['total_investment'] += $investment;

        $status = $project['status'] ?? 'Unknown';
        $dashboard_data['status'][$status] = ($dashboard_data['status'][$status] ?? 0) + 1;

        $sector = $project['pillar_title'] ?? 'Uncategorized';
        $dashboard_data['sector'][$sector] = ($dashboard_data['sector'][$sector] ?? 0) + 1;

        if ($ward_id) { $location_name = $project['ward_name']; }
        elseif ($constituency_id) { $location_name = $project['ward_name']; }
        else { $location_name = $project['constituency_name']; }
        $dashboard_data['location'][$location_name ?? 'N/A'] = ($dashboard_data['location'][$location_name ?? 'N/A'] ?? 0) + 1;

        $allocated = (float)($project['budget_allocated'] ?? 0);
        $spent = (float)($project['budget_spent'] ?? 0);
        $dashboard_data['financials']['allocated'] += $allocated;
        $dashboard_data['financials']['spent'] += $spent;

        if ($project['status'] === 'Completed' && !empty($project['expected_completion_date']) && !empty($project['actual_completion_date'])) {
            if ($project['actual_completion_date'] <= $project['expected_completion_date']) {
                $dashboard_data['timeliness']['On Time']++;
            } else {
                $dashboard_data['timeliness']['Late']++;
                $delay = (new DateTime($project['expected_completion_date']))->diff(new DateTime($project['actual_completion_date']))->days;
                $total_delay += $delay;
                $late_projects_count++;
            }
        }
        
        if (!isset($dashboard_data['budgetUtilization'][$sector])) {
            $dashboard_data['budgetUtilization'][$sector] = ['allocated' => 0, 'spent' => 0];
        }
        $dashboard_data['budgetUtilization'][$sector]['allocated'] += $allocated;
        $dashboard_data['budgetUtilization'][$sector]['spent'] += $spent;
    }

    if ($late_projects_count > 0) {
        $dashboard_data['kpis']['avg_delay'] = round($total_delay / $late_projects_count);
    }
    arsort($dashboard_data['location']);

    echo json_encode($dashboard_data);

} catch (PDOException $e) {
    http_response_code(500);
    error_log("Dashboard API Error: " . $e->getMessage());
    echo json_encode(['error' => 'Database error occurred: ' . $e->getMessage()]);
}