<?php
// FILE: admin/view_knowledge.php
// PURPOSE: Displays a document from the knowledge base, its full extracted text, and its individual text chunks.

// --- Security & Session Management ---
require_once '../functions.php'; // Contains helper functions like require_login(), is_admin() etc.
require_once '../db_connect.php';   // This handles the database connection ($pdo).

// Ensure the user is logged in.
require_login(); 
// Ensure the user has the correct role to access this page.
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php"); // Redirect non-admins
    exit;
}

// --- CSRF Protection ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}


// Sanitize the input ID
$doc_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$doc_id) {
    http_response_code(400); // Bad Request
    die("Error: A valid document ID is required.");
}

// Fetch the main document details
$stmt = $pdo->prepare("SELECT * FROM knowledge_base WHERE id = ?");
$stmt->execute([$doc_id]);
$doc = $stmt->fetch();

if (!$doc) {
    http_response_code(404); // Not Found
    die("Error: The requested document could not be found.");
}

// Fetch all chunks for this document
$stmt_chunks = $pdo->prepare("SELECT * FROM knowledge_base_chunks WHERE knowledge_base_id = ? ORDER BY chunk_index ASC");
$stmt_chunks->execute([$doc_id]);
$chunks = $stmt_chunks->fetchAll();

// Get file size for display
$filePath = '../' . $doc['file_path'];
$fileSize = file_exists($filePath) ? round(filesize($filePath) / 1024, 2) . ' KB' : 'N/A';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>View Document: <?= htmlspecialchars($doc['title']) ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 font-sans">
    <div class="container mx-auto p-4 sm:p-8">
        <div class="max-w-5xl mx-auto">
            <a href="upload_knowledge.php" class="text-blue-600 hover:underline mb-6 inline-block">&larr; Back to Knowledge Base</a>
            
            <div class="bg-white p-6 sm:p-8 rounded-2xl shadow-lg">
                <!-- Document Metadata Section -->
                <div class="border-b border-gray-200 pb-6 mb-6">
                    <h1 class="text-3xl font-bold text-gray-800 mb-3"><?= htmlspecialchars($doc['title']) ?></h1>
                    <div class="flex flex-wrap text-sm text-gray-600 gap-x-6 gap-y-2">
                        <span><strong>Type:</strong> <?= htmlspecialchars($doc['document_type']) ?></span>
                        <span><strong>Uploaded:</strong> <?= date("F j, Y, g:i a", strtotime($doc['uploaded_at'])) ?></span>
                        <span><strong>File Size:</strong> <?= htmlspecialchars($fileSize) ?></span>
                        <span><strong>File:</strong> <a href="../<?= htmlspecialchars($doc['file_path']) ?>" class="text-blue-600 underline" target="_blank" rel="noopener noreferrer">Download Original</a></span>
                    </div>
                </div>

                <!-- Full Extracted Text Section -->
                <div class="mb-10">
                    <h2 class="text-2xl font-semibold text-gray-800 mb-4">Full Extracted Text</h2>
                    <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg max-h-96 overflow-y-auto text-gray-700 text-sm leading-relaxed">
                        <pre class="whitespace-pre-wrap font-sans"><?= htmlspecialchars($doc['content_text']) ?></pre>
                    </div>
                </div>

                <!-- Text Chunks Section -->
                <div>
                    <h2 class="text-2xl font-semibold text-gray-800 mb-4">Searchable Text Chunks (<?= count($chunks) ?>)</h2>
                    
                    <?php if (empty($chunks)): ?>
                        <div class="bg-yellow-100 text-yellow-800 p-4 rounded-lg">
                            <p>No text chunks were generated for this document. This might happen if the document was empty or the text could not be extracted.</p>
                        </div>
                    <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($chunks as $chunk): ?>
                                <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg transition hover:shadow-sm hover:border-blue-300">
                                    <h3 class="font-bold text-gray-900 mb-2">Chunk #<?= $chunk['chunk_index'] + 1 ?></h3>
                                    <p class="text-gray-700 whitespace-pre-wrap text-sm leading-relaxed"><?= htmlspecialchars($chunk['content_text']) ?></p>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
