<?php
require_once 'functions.php';
require_once 'db.php';

$errors = [];
$success_message = '';
$inputs = [];
try {
    $wards = $pdo->query("SELECT id, name FROM wards ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db'] = "Could not fetch ward list. Please contact an administrator.";
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitize and store inputs to re-populate the form on error
    $inputs['name'] = trim($_POST['name']);
    $inputs['email'] = trim($_POST['email']);
    $inputs['ward_id'] = trim($_POST['ward_id']);
    $password = trim($_POST['password']);
    $password_confirm = trim($_POST['password_confirm']);
    
    // --- Validation ---
    if (empty($inputs['name'])) $errors['name'] = "Full name is required.";
    if (empty($inputs['email'])) {
        $errors['email'] = "Email is required.";
    } elseif (!filter_var($inputs['email'], FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = "Please enter a valid email address.";
    }

    if (empty($inputs['ward_id'])) $errors['ward_id'] = "Please select your assigned ward.";
    if (empty($password)) $errors['password'] = "Password is required.";
    if (strlen($password) < 8) $errors['password'] = "Password must be at least 8 characters long.";
    if ($password !== $password_confirm) $errors['password_confirm'] = "Passwords do not match.";

    // --- Check if email already exists ---
    if (empty($errors['email'])) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :email");
            $stmt->execute(['email' => $inputs['email']]);
            if ($stmt->fetch()) {
                $errors['email'] = "This email address is already registered.";
            }
        } catch (PDOException $e) {
            $errors['db'] = "A database error occurred during validation.";
        }
    }

    // --- Process registration if no errors ---
    if (empty($errors)) {
        try {
            // Hash the password for security
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Prepare SQL to insert new user
            $sql = "INSERT INTO users (name, email, password, role, ward_id, is_active) VALUES (:name, :email, :password, 'Inputer', :ward_id, 0)";
            $stmt = $pdo->prepare($sql);
            
            $stmt->execute([
                'name' => $inputs['name'],
                'email' => $inputs['email'],
                'password' => $hashed_password,
                'ward_id' => $inputs['ward_id']
            ]);

            // Set success message and clear inputs
            $success_message = "Registration successful! Your account is now pending approval from an administrator. You will be unable to log in until your account has been activated.";
            $inputs = [];

        } catch (PDOException $e) {
            $errors['db'] = "An error occurred during registration. Please try again. Error: " . $e->getMessage();
        }
    }
}


include 'includes/header.php';
?>

<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        
        <div>
            <!-- UPDATED STANDARD LOGO -->
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Create New Account
            </h2>
            <p class="mt-2 text-center text-sm text-gray-600">
                For Ward Administrators & Project Inputers
            </p>
        </div>

        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert">
                    <p class="font-bold">Success</p>
                    <p><?php echo e($success_message); ?></p>
                </div>
            <?php endif; ?>

            <?php if (isset($errors['db'])): ?>
                 <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                    <p><?php echo e($errors['db']); ?></p>
                </div>
            <?php endif; ?>

            <form class="space-y-6" action="signup.php" method="post" novalidate>
                <div>
                    <label for="name" class="block text-sm font-medium text-gray-700">Full Name</label>
                    <input id="name" name="name" type="text" required class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="John Doe" value="<?php echo e($inputs['name'] ?? ''); ?>">
                    <?php if(isset($errors['name'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['name']; ?></p><?php endif; ?>
                </div>

                 <div>
                    <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                    <input id="email" name="email" type="email" autocomplete="email" required class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="you@example.com" value="<?php echo e($inputs['email'] ?? ''); ?>">
                    <?php if(isset($errors['email'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['email']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="ward_id" class="block text-sm font-medium text-gray-700">Assigned Ward</label>
                    <select name="ward_id" id="ward_id" required class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 bg-white text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm">
                        <option value="">-- Select Your Ward --</option>
                        <?php if (!empty($wards)): ?>
                            <?php foreach ($wards as $ward): ?>
                                <option value="<?php echo e($ward['id']); ?>" <?php echo (($inputs['ward_id'] ?? '') == $ward['id']) ? 'selected' : ''; ?>>
                                    <?php echo e($ward['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </select>
                     <?php if(isset($errors['ward_id'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['ward_id']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="password" class="block text-sm font-medium text-gray-700">Password</label>
                    <input id="password" name="password" type="password" required class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Minimum 8 characters">
                    <?php if(isset($errors['password'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password']; ?></p><?php endif; ?>
                </div>

                 <div>
                    <label for="password_confirm" class="block text-sm font-medium text-gray-700">Confirm Password</label>
                    <input id="password_confirm" name="password_confirm" type="password" required class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Re-enter your password">
                    <?php if(isset($errors['password_confirm'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password_confirm']; ?></p><?php endif; ?>
                </div>
    
                <div>
                    <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out">
                        Register
                    </button>
                </div>
            </form>
            <p class="text-center text-sm text-gray-600">
                Already have an account? 
                <a href="login.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign in here
                </a>
            </p>
        </div>
    </div>
</div>

<?php
include 'includes/footer.php';
?>
