<?php
// FILE: admin/media_library.php
// PURPOSE: A central gallery to upload, view, and manage all site images.

require_once 'functions.php';
require_once 'db.php';

require_login();

$feedback = [];
$user_can_delete = is_super_admin() || is_admin();

// --- Handle Multiple File Uploads ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES['image_uploads'])) {
    $upload_dir = 'uploads/';
    if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
    
    $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $success_count = 0;
    $error_count = 0;

    foreach ($_FILES['image_uploads']['name'] as $key => $name) {
        if ($_FILES['image_uploads']['error'][$key] === UPLOAD_ERR_OK) {
            $tmp_name = $_FILES['image_uploads']['tmp_name'][$key];
            $file_type = $_FILES['image_uploads']['type'][$key];
            $file_size = $_FILES['image_uploads']['size'][$key];

            if (!in_array($file_type, $allowed_types) || $file_size > 5000000) {
                $error_count++;
                continue; 
            }

            $file_name = 'media_' . uniqid() . '.' . strtolower(pathinfo($name, PATHINFO_EXTENSION));
            $destination = $upload_dir . $file_name;

            if (move_uploaded_file($tmp_name, $destination)) {
                try {
                    $sql = "INSERT INTO media_gallery (file_name, file_type, file_size, alt_text, uploaded_by) VALUES (:file_name, :file_type, :file_size, :alt_text, :uploaded_by)";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([
                        'file_name' => $file_name,
                        'file_type' => $file_type,
                        'file_size' => $file_size,
                        'alt_text' => pathinfo($name, PATHINFO_FILENAME), // Use original name as default alt text
                        'uploaded_by' => $_SESSION['id']
                    ]);
                    $success_count++;
                } catch (PDOException $e) {
                    $error_count++;
                    unlink($destination); 
                }
            } else {
                $error_count++;
            }
        }
    }
    
    if($success_count > 0) {
        log_action($pdo, "Uploaded {$success_count} images", "MediaGallery", $_SESSION['id']);
    }
    $feedback['message'] = "Upload complete. {$success_count} files uploaded successfully.";
    if ($error_count > 0) {
        $feedback['message'] .= " {$error_count} files failed due to size or type errors.";
    }
    $feedback['type'] = 'success';
}


// --- Setup Pagination ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 24; 
$offset = ($page - 1) * $limit;

$images = [];
$total_images = 0;

try {
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM media_gallery");
    $total_images = $count_stmt->fetchColumn();
    $total_pages = ceil($total_images / $limit);

    $sql = "SELECT m.*, u.name as uploader_name FROM media_gallery m JOIN users u ON m.uploaded_by = u.id ORDER BY m.uploaded_at DESC LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $images = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching media: " . $e->getMessage();
}


include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8 flex justify-between items-center">
        <div>
            <h1 class="text-lg font-semibold leading-6 text-gray-900">Media Gallery</h1>
            <p class="mt-1 text-sm text-gray-500">Total images: <?php echo e(number_format($total_images)); ?></p>
        </div>
         <button id="add-new-btn" class="inline-flex items-center gap-x-2 rounded-md bg-brand-orange px-3.5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-orange-500">
            Add New Media
        </button>
    </div>
</header>
    
<div class="mt-6">
     <?php if (!empty($feedback)): ?>
        <div class="mb-4 p-4 rounded-md <?php echo $feedback['type'] === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
            <?php echo e($feedback['message']); ?>
        </div>
    <?php endif; ?>
    <div class="grid grid-cols-2 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8 gap-4">
        <?php foreach($images as $image): ?>
            <div class="relative group cursor-pointer" onclick="openEditModal(<?php echo htmlspecialchars(json_encode($image), ENT_QUOTES, 'UTF-8'); ?>)">
                <div class="aspect-square w-full overflow-hidden rounded-lg bg-gray-100">
                     <img src="uploads/<?php echo e($image['file_name']); ?>" alt="<?php echo e($image['alt_text']); ?>" class="w-full h-full object-cover">
                </div>
                 <div class="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                    <p class="text-white text-sm font-bold">Details</p>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    <div class="text-sm text-gray-700">Page <?php echo $page; ?> of <?php echo $total_pages; ?></div>
    <?php if ($page < $total_pages): ?>
        <a href="?page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<!-- Upload Modal -->
<div id="upload-modal" class="fixed inset-0 bg-gray-600 bg-opacity-75 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center border-b pb-3 mb-4">
            <h3 class="text-lg font-medium text-gray-900">Upload New Media</h3>
            <button id="close-upload-modal" class="text-gray-400 hover:text-gray-600">&times;</button>
        </div>
        <form action="media_library.php" method="post" enctype="multipart/form-data">
            <div>
                <label class="block text-sm font-medium text-gray-900" for="image_uploads">Select Images (Multiple allowed)</label>
                <p class="text-xs text-gray-500 mb-2">Max 5MB per file. Allowed types: JPG, PNG, GIF, WEBP.</p>
                <input type="file" name="image_uploads[]" id="image_uploads" multiple required class="block w-full text-sm text-slate-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-violet-50 file:text-brand-purple hover:file:bg-violet-100">
            </div>
            <div class="mt-6 flex justify-end">
                <button type="submit" class="rounded-md bg-brand-orange px-4 py-2 text-sm font-semibold text-white shadow-sm hover:bg-opacity-90">Upload Files</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Modal -->
<div id="edit-modal" class="fixed inset-0 bg-gray-600 bg-opacity-75 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-10 mx-auto p-5 border w-full max-w-4xl shadow-lg rounded-md bg-white">
         <div class="flex justify-between items-center border-b pb-3 mb-4">
            <h3 class="text-lg font-medium text-gray-900">Attachment Details</h3>
            <button id="close-edit-modal" class="text-gray-400 hover:text-gray-600">&times;</button>
        </div>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div class="md:col-span-2">
                <img id="edit-preview-img" src="" class="w-full h-auto max-h-[60vh] object-contain rounded-md bg-gray-100">
            </div>
            <div class="space-y-4">
                 <div>
                    <label for="edit-alt-text" class="block text-sm font-medium text-gray-700">Alt Text (Caption)</label>
                    <input type="text" id="edit-alt-text" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm py-2 px-3">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700">File URL</label>
                    <input type="text" id="edit-file-url" readonly class="mt-1 block w-full rounded-md border-gray-300 shadow-sm py-2 px-3 bg-gray-100">
                    <button id="copy-url-btn" class="mt-1 text-sm text-brand-purple hover:underline">Copy URL to clipboard</button>
                </div>
                 <div>
                    <p class="text-xs text-gray-500"><strong>Uploaded:</strong> <span id="edit-uploaded-at"></span></p>
                    <p class="text-xs text-gray-500"><strong>By:</strong> <span id="edit-uploader"></span></p>
                    <p class="text-xs text-gray-500"><strong>Size:</strong> <span id="edit-size"></span></p>
                </div>
                <div class="flex items-center justify-between pt-4 border-t">
                    <button id="delete-btn" class="text-sm text-red-600 hover:underline" <?php if (!$user_can_delete) echo 'style="display:none;"'; ?>>Delete Permanently</button>
                    <button id="update-details-btn" class="rounded-md bg-brand-blue px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-opacity-90">Update</button>
                </div>
                 <form id="delete-form" action="delete_media.php" method="post" style="display:none;"><input type="hidden" name="image_id" id="delete-image-id"></form>
            </div>
        </div>
    </div>
</div>


<script>
document.addEventListener('DOMContentLoaded', function() {
    const uploadModal = document.getElementById('upload-modal');
    const addNewBtn = document.getElementById('add-new-btn');
    const closeUploadModalBtn = document.getElementById('close-upload-modal');

    addNewBtn.addEventListener('click', () => uploadModal.style.display = 'block');
    closeUploadModalBtn.addEventListener('click', () => uploadModal.style.display = 'none');

    const editModal = document.getElementById('edit-modal');
    const closeEditModalBtn = document.getElementById('close-edit-modal');
    closeEditModalBtn.addEventListener('click', () => editModal.style.display = 'none');

    window.openEditModal = function(image) {
        document.getElementById('edit-preview-img').src = 'uploads/' + image.file_name;
        document.getElementById('edit-alt-text').value = image.alt_text || '';
        const fileUrl = window.location.origin + '/admin/uploads/' + image.file_name;
        document.getElementById('edit-file-url').value = fileUrl;
        document.getElementById('edit-uploaded-at').textContent = new Date(image.uploaded_at).toLocaleDateString();
        document.getElementById('edit-uploader').textContent = image.uploader_name;
        document.getElementById('edit-size').textContent = (image.file_size / 1024).toFixed(2) + ' KB';
        document.getElementById('delete-image-id').value = image.id;
        
        const copyBtn = document.getElementById('copy-url-btn');
        copyBtn.onclick = () => {
            navigator.clipboard.writeText(fileUrl).then(() => {
                copyBtn.textContent = 'Copied!';
                setTimeout(() => { copyBtn.textContent = 'Copy URL to clipboard'; }, 2000);
            });
        };
        
        const deleteBtn = document.getElementById('delete-btn');
        deleteBtn.onclick = () => {
            if (confirm('Are you sure you want to permanently delete this image? This action cannot be undone.')) {
                document.getElementById('delete-form').submit();
            }
        };

        editModal.style.display = 'block';
    }
});
</script>

<?php include 'includes/footer.php'; ?>
