<?php
// FILE: admin/gallery_dashboard.php
// PURPOSE: To view and manage all media galleries.

require_once 'functions.php';
require_once 'db.php';

require_login();

// --- Handle New Gallery Creation ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['create_gallery'])) {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    if (!empty($title)) {
        try {
            $sql = "INSERT INTO galleries (title, description, created_by) VALUES (:title, :description, :created_by)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(['title' => $title, 'description' => $description, 'created_by' => $_SESSION['id']]);
            log_action($pdo, "Created Gallery", "Gallery", $pdo->lastInsertId());
            header("Location: gallery_dashboard.php?success=created");
            exit;
        } catch (PDOException $e) {
            $db_error = "Error creating gallery: " . $e->getMessage();
        }
    } else {
        $form_error = "Gallery title is required.";
    }
}

// --- Fetch all galleries ---
$galleries = [];
try {
    $stmt = $pdo->query("
        SELECT g.*, u.name as author, 
               (SELECT COUNT(id) FROM media_gallery WHERE gallery_id = g.id) as image_count,
               (SELECT file_name FROM media_gallery WHERE gallery_id = g.id ORDER BY id DESC LIMIT 1) as cover_image
        FROM galleries g
        JOIN users u ON g.created_by = u.id
        ORDER BY g.created_at DESC
    ");
    $galleries = $stmt->fetchAll();
} catch (PDOException $e) {
    $db_error = "Error fetching galleries: " . $e->getMessage();
}

include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8 flex justify-between items-center">
        <div>
            <h1 class="text-lg font-semibold leading-6 text-gray-900">Media Galleries</h1>
            <p class="mt-1 text-sm text-gray-500">Organize your images into galleries.</p>
        </div>
        <button id="add-new-btn" class="inline-flex items-center gap-x-2 rounded-md bg-brand-orange px-3.5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-orange-500">
            Create New Gallery
        </button>
    </div>
</header>
    
<div class="mt-6">
    <?php if (!empty($form_error)): ?>
        <div class="mb-4 p-4 rounded-md bg-red-100 text-red-700"><?php echo e($form_error); ?></div>
    <?php endif; ?>
    <?php if (!empty($db_error)): ?>
        <div class="mb-4 p-4 rounded-md bg-red-100 text-red-700"><?php echo e($db_error); ?></div>
    <?php endif; ?>

    <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
        <?php foreach($galleries as $gallery): ?>
        <a href="manage_gallery.php?id=<?php echo e($gallery['id']); ?>" class="group relative aspect-w-1 aspect-h-1 block w-full overflow-hidden rounded-lg bg-gray-100 focus-within:ring-2 focus-within:ring-brand-purple focus-within:ring-offset-2 focus-within:ring-offset-gray-100">
            <img src="uploads/<?php echo e($gallery['cover_image'] ?? 'placeholder.jpg'); ?>" alt="" class="pointer-events-none object-cover group-hover:opacity-75" onerror="this.onerror=null;this.src='https://placehold.co/600x400/1E3A8A/FFFFFF?text=No+Images';">
            <div class="absolute inset-0 bg-black/40"></div>
            <div class="absolute inset-0 flex flex-col justify-end p-4 text-white">
                <p class="font-bold truncate"><?php echo e($gallery['title']); ?></p>
                <p class="text-sm"><?php echo e($gallery['image_count']); ?> images</p>
            </div>
        </a>
        <?php endforeach; ?>
    </div>
</div>

<!-- Add New Gallery Modal -->
<div id="new-gallery-modal" class="fixed inset-0 bg-gray-600 bg-opacity-75 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center border-b pb-3 mb-4">
            <h3 class="text-lg font-medium text-gray-900">Create a New Gallery</h3>
            <button id="close-modal-btn" class="text-gray-400 hover:text-gray-600">&times;</button>
        </div>
        <form action="gallery_dashboard.php" method="post" class="space-y-4">
            <div>
                <label for="title" class="block text-sm font-medium text-gray-700">Gallery Title</label>
                <input type="text" name="title" id="title" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm py-2 px-3">
            </div>
            <div>
                <label for="description" class="block text-sm font-medium text-gray-700">Description (Optional)</label>
                <textarea name="description" id="description" rows="3" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm py-2 px-3"></textarea>
            </div>
            <div class="mt-6 flex justify-end">
                <button type="submit" name="create_gallery" class="rounded-md bg-brand-orange px-4 py-2 text-sm font-semibold text-white shadow-sm hover:bg-opacity-90">Create Gallery</button>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('new-gallery-modal');
    const addBtn = document.getElementById('add-new-btn');
    const closeBtn = document.getElementById('close-modal-btn');

    addBtn.addEventListener('click', () => modal.style.display = 'block');
    closeBtn.addEventListener('click', () => modal.style.display = 'none');
    window.addEventListener('click', (event) => {
        if (event.target == modal) {
            modal.style.display = 'none';
        }
    });
});
</script>

<?php include 'includes/footer.php'; ?>
