<?php
require_once 'functions.php';
require_once 'db.php';
require_login();

$page = max(1, intval($_GET['page'] ?? 1));
$page_size = max(1, min(50, intval($_GET['page_size'] ?? 10)));
$offset = ($page - 1) * $page_size;
$search = trim($_GET['search'] ?? '');
$status = $_GET['status'] ?? 'All';
$export = isset($_GET['export']);
$is_inputer = is_inputer();
$user_id = $_SESSION['id'];

$where = "p.is_approved = 1";
$params = [];

if ($is_inputer) {
    $where .= " AND p.created_by = :uid";
    $params['uid'] = $user_id;
}
if ($status && $status !== 'All') {
    $where .= " AND p.status = :status";
    $params['status'] = $status;
}
if ($search !== '') {
    // Case-insensitive search for project name, ward, constituency, or status
    $where .= " AND (LOWER(p.name) LIKE :search OR LOWER(w.name) LIKE :search OR LOWER(c.name) LIKE :search OR LOWER(p.status) LIKE :search)";
    $params['search'] = '%' . strtolower($search) . '%';
}

// Main paged query
$sql = "SELECT p.id, p.name as project_name, p.status, w.name as ward_name, c.name as constituency_name
        FROM projects p
        JOIN wards w ON p.ward_id = w.id
        JOIN constituencies c ON w.constituency_id = c.id
        WHERE $where
        ORDER BY p.updated_at DESC
        LIMIT :offset, :limit";

$stmt = $pdo->prepare($sql);
foreach ($params as $k => $v) {
    $type = is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR;
    $stmt->bindValue(':' . $k, $v, $type);
}
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':limit', $page_size, PDO::PARAM_INT);
$stmt->execute();
$projects = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Total count for pagination
$count_sql = "SELECT COUNT(*) FROM projects p JOIN wards w ON p.ward_id = w.id JOIN constituencies c ON w.constituency_id = c.id WHERE $where";
$count_stmt = $pdo->prepare($count_sql);
foreach ($params as $k => $v) {
    $type = is_int($v) ? PDO::PARAM_INT : PDO::PARAM_STR;
    $count_stmt->bindValue(':' . $k, $v, $type);
}
$count_stmt->execute();
$total = $count_stmt->fetchColumn();

// KPIs for dashboard
$kpis = ['total' => 0, 'ongoing' => 0, 'completed' => 0, 'planned' => 0, 'stalled' => 0];
$kpi_sql = "SELECT status, COUNT(*) as cnt FROM projects WHERE is_approved = 1 GROUP BY status";
foreach ($pdo->query($kpi_sql) as $row) {
    $kpis[strtolower($row['status'])] = (int)$row['cnt'];
    $kpis[($row['status'])] = (int)$row['cnt'];
    $kpis['total'] += (int)$row['cnt'];
}

// Bar graph: Projects by Constituency (regardless of search/filter, for the dashboard)
$by_constituency = [];
$bar_sql = "SELECT c.name as constituency, COUNT(p.id) as total
            FROM projects p
            JOIN wards w ON p.ward_id = w.id
            JOIN constituencies c ON w.constituency_id = c.id
            WHERE p.is_approved = 1
            GROUP BY c.name
            ORDER BY total DESC";
foreach ($pdo->query($bar_sql) as $row) {
    $by_constituency[] = [
        'constituency' => $row['constituency'],
        'total' => (int)$row['total'],
    ];
}

// CSV export (download)
if ($export) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="approved_projects.csv"');
    $f = fopen('php://output', 'w');
    fputcsv($f, ['Project', 'Ward', 'Constituency', 'Status']);
    foreach ($projects as $p) fputcsv($f, [$p['project_name'], $p['ward_name'], $p['constituency_name'], $p['status']]);
    exit;
}

// Pending section as HTML (for dashboard)
if (isset($_GET['pending'])) {
    $out = '';
    if (is_super_admin() || is_admin() || is_approver()) {
        $pending = $pdo->query("SELECT p.id, p.name as project_name, w.name as ward_name, u.name as created_by_name 
            FROM projects p
            JOIN wards w ON p.ward_id = w.id
            JOIN users u ON p.created_by = u.id
            WHERE p.is_approved = 0 ORDER BY p.created_at DESC")->fetchAll();
        if ($pending) {
            $out .= '<div id="pending" class="mb-8"><div class="bg-yellow-50 border border-yellow-200 rounded-lg shadow-sm">';
            $out .= '<div class="p-4 border-b border-yellow-200"><h3 class="text-lg font-medium leading-6 text-yellow-900">Projects Pending Approval</h3></div><ul class="divide-y divide-yellow-200">';
            foreach ($pending as $p) {
                $out .= '<li class="flex items-center justify-between gap-x-6 px-4 py-3">';
                $out .= '<div class="min-w-0"><p class="text-sm font-semibold leading-6 text-gray-900">'.e($p['project_name']).'</p><p class="mt-1 truncate text-xs leading-5 text-gray-500">Submitted by '.e($p['created_by_name']).' for '.e($p['ward_name']).' Ward</p></div>';
                $out .= '<a href="manage_project.php?id='.e($p['id']).'" class="rounded-full bg-white px-2.5 py-1 text-xs font-semibold text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 hover:bg-gray-50">Review & Approve</a>';
                $out .= '</li>';
            }
            $out .= '</ul></div></div>';
        }
    }
    echo $out;
    exit;
}

// Return all dashboard data as JSON
echo json_encode([
    'projects' => $projects,
    'total' => (int)$total,
    'kpis' => $kpis,
    'by_constituency' => $by_constituency
]);
