<?php
// SETS THE DEFAULT TIMEZONE FOR THE ENTIRE APPLICATION
date_default_timezone_set('Africa/Nairobi');

require_once 'functions.php';
require_once 'db.php';

// Security: Only SuperAdmins can access this page
require_login();
require_super_admin();

// --- Fetch data for form dropdowns ---
$wards = [];
$constituencies = [];
$pillars = [];
try {
    $wards = $pdo->query("SELECT id, name, constituency_id FROM wards ORDER BY name ASC")->fetchAll();
    $constituencies = $pdo->query("SELECT id, name FROM constituencies ORDER BY name ASC")->fetchAll();
    $pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY id ASC")->fetchAll();
} catch (PDOException $e) {
    $db_error = "Could not fetch form data: " . $e->getMessage();
}

$report_data_json = null;
$report_title = "Report Preview";
$report_period = "All Time";

// --- Check for form submission and process data ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['generate_report'])) {
    
    // --- Get form data ---
    $report_level = $_POST['report_level'] ?? 'county';
    $location_id = $_POST['location_id'] ?? null;
    $pillar_id = $_POST['pillar_id'] ?? null;
    $start_date = $_POST['start_date'] ?? null;
    $end_date = $_POST['end_date'] ?? null;

    if (empty($location_id) && $report_level != 'county') {
        $form_error = "Please select a location for this report level.";
    } else {
        try {
            // --- Build Dynamic SQL Query ---
            $sql_base = "FROM projects p 
                    JOIN wards w ON p.ward_id = w.id
                    JOIN constituencies c ON w.constituency_id = c.id";
            
            $where_clauses = [];
            $params = [];
            $report_title = "County-Wide";

            if ($report_level == 'constituency') {
                $where_clauses[] = "c.id = :location_id";
                $params['location_id'] = $location_id;
                $report_title = $constituencies[array_search($location_id, array_column($constituencies, 'id'))]['name'] . " Constituency";
            } elseif ($report_level == 'ward') {
                $where_clauses[] = "w.id = :location_id";
                $params['location_id'] = $location_id;
                $report_title = $wards[array_search($location_id, array_column($wards, 'id'))]['name'] . " Ward";
            }

            if (!empty($pillar_id)) {
                $where_clauses[] = "p.manifesto_pillar_id = :pillar_id";
                $params['pillar_id'] = $pillar_id;
                $report_title .= " (" . $pillars[array_search($pillar_id, array_column($pillars, 'id'))]['title'] . ")";
            }
            if (!empty($start_date)) { 
                $where_clauses[] = "p.start_date >= :start_date"; 
                $params['start_date'] = $start_date; 
                $report_period = date("M d, Y", strtotime($start_date)) . " to ";
            }
            if (!empty($end_date)) { 
                $where_clauses[] = "p.start_date <= :end_date"; 
                $params['end_date'] = $end_date; 
                $report_period .= date("M d, Y", strtotime($end_date));
            } else {
                 if(!empty($start_date)) $report_period .= "Present";
            }

            $sql_where = !empty($where_clauses) ? " WHERE " . implode(" AND ", $where_clauses) : "";

            // --- Fetch all required data in one go ---
            $queries = [
                'kpis' => "SELECT COUNT(p.id) as total_projects, SUM(p.budget_allocated) as total_investment, SUM(CASE WHEN p.status = 'Completed' THEN 1 ELSE 0 END) as completed_projects, SUM(CASE WHEN p.status = 'Ongoing' THEN 1 ELSE 0 END) as ongoing_projects $sql_base $sql_where",
                'status' => "SELECT p.status, COUNT(p.id) as count $sql_base $sql_where GROUP BY p.status",
                'sector' => "SELECT mp.title, COUNT(p.id) as count FROM projects p JOIN manifesto_pillars mp ON p.manifesto_pillar_id = mp.id JOIN wards w ON p.ward_id = w.id JOIN constituencies c ON w.constituency_id = c.id $sql_where GROUP BY mp.title",
                'location' => $report_level == 'county' 
                    ? "SELECT c.name, COUNT(p.id) as count $sql_base $sql_where GROUP BY c.name ORDER BY count DESC" 
                    : "SELECT w.name, COUNT(p.id) as count $sql_base $sql_where GROUP BY w.name ORDER BY count DESC",
                'financial_burn' => "SELECT DATE_FORMAT(p.start_date, '%Y-%m') as month, SUM(p.budget_allocated) as allocated, SUM(p.budget_spent) as spent $sql_base $sql_where AND p.start_date IS NOT NULL GROUP BY month ORDER BY month ASC",
                'pillar_status' => "SELECT mp.title, p.status, COUNT(p.id) as count FROM projects p JOIN manifesto_pillars mp ON p.manifesto_pillar_id = mp.id JOIN wards w ON p.ward_id = w.id JOIN constituencies c ON w.constituency_id = c.id $sql_where GROUP BY mp.title, p.status"
            ];

            $report_data = [];
            foreach ($queries as $key => $query) {
                $stmt = $pdo->prepare($query);
                $stmt->execute($params);
                if ($key === 'kpis') {
                    $report_data[$key] = $stmt->fetch(PDO::FETCH_ASSOC);
                } elseif (in_array($key, ['status', 'sector', 'location'])) {
                    $report_data[$key] = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
                } else {
                     $report_data[$key] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                }
            }
            
            if (empty($report_data['kpis']['total_projects'])) {
                $form_error = "No projects found for the selected criteria.";
            } else {
                 // NEW: Log the action after data is successfully fetched
                 log_action($pdo, 'Generated Visual Report', 'Report', intval($location_id));
                 $report_data_json = json_encode($report_data);
            }

        } catch (PDOException $e) {
            $db_error = "Error generating report: " . $e->getMessage();
        }
    }
}

include 'includes/header.php';
?>

<!-- Include Chart.js and html2canvas libraries -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>

<div class="min-h-full">
    <header class="bg-white shadow">
        <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8">
            <h1 class="text-3xl font-bold tracking-tight text-gray-900">Generate Visual Report</h1>
        </div>
    </header>

    <main class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
        <!-- Form Section -->
        <div class="bg-white p-6 rounded-lg shadow-lg mb-8">
            <form action="visual_report.php" method="post" class="space-y-4">
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                     <div>
                        <label for="report_level" class="block text-sm font-medium text-gray-700">Report Level</label>
                        <select name="report_level" id="report_level" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm px-3 py-2">
                            <option value="county" <?php if(($_POST['report_level'] ?? '') == 'county') echo 'selected'; ?>>County-Wide</option>
                            <option value="constituency" <?php if(($_POST['report_level'] ?? '') == 'constituency') echo 'selected'; ?>>Constituency</option>
                            <option value="ward" <?php if(($_POST['report_level'] ?? '') == 'ward') echo 'selected'; ?>>Ward</option>
                        </select>
                    </div>
                    <div>
                        <label for="location_id" class="block text-sm font-medium text-gray-700">Location</label>
                        <select name="location_id" id="location_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm px-3 py-2"> </select>
                    </div>
                     <div>
                        <label for="pillar_id" class="block text-sm font-medium text-gray-700">Sector (Optional)</label>
                        <select name="pillar_id" id="pillar_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm px-3 py-2">
                            <option value="">All Sectors</option>
                            <?php foreach ($pillars as $pillar): ?>
                                <option value="<?php echo e($pillar['id']); ?>" <?php if(($_POST['pillar_id'] ?? '') == $pillar['id']) echo 'selected'; ?>><?php echo e($pillar['title']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                 <div class="flex justify-end">
                    <button type="submit" name="generate_report" class="inline-flex justify-center rounded-md border border-transparent bg-brand-blue py-2 px-4 text-sm font-medium text-white shadow-sm hover:bg-opacity-90">Generate Report</button>
                </div>
            </form>
        </div>
        
        <!-- Preview and Download Section -->
        <div class="bg-gray-200 p-6 rounded-lg shadow-inner">
             <div class="flex justify-between items-center mb-4">
                <h2 class="text-xl font-bold text-gray-800">Report Preview</h2>
                <button id="download-btn" class="inline-flex items-center rounded-md border border-transparent bg-brand-orange px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-opacity-90 disabled:bg-gray-400" disabled>Download as PNG</button>
            </div>
            
            <!-- This is the container that will be screenshotted -->
            <div id="report-preview" class="p-8 border bg-white font-sans mx-auto" style="width: 800px; height: 1120px;">
                <!-- Report Header -->
                <div class="flex items-start justify-between border-b-2 border-brand-blue pb-4 mb-4">
                    <div class="flex items-center space-x-3"><img src="assets/hbc_logo.png" class="h-20 w-auto"><div class="h-16 w-px bg-gray-300"></div><div><p class="font-extrabold text-3xl text-brand-blue">GENOWA</p><p class="block font-semibold text-base text-brand-orange -mt-1">DELIVERY UNIT</p></div></div>
                    <div class="text-right"><p class="text-xl font-bold text-gray-800">Project Delivery Summary</p><p id="report-title-preview" class="text-lg text-gray-600"><?php echo e($report_title); ?></p><p class="text-xs text-gray-500">For Period: <?php echo e($report_period); ?></p></div>
                </div>
                <!-- KPIs -->
                <div id="kpi-container" class="grid grid-cols-4 gap-4 text-center mb-8"></div>
                <!-- Charts Row 1 -->
                <div class="grid grid-cols-5 gap-6 mb-6 h-52">
                    <div class="col-span-2 space-y-4">
                        <div class="text-center"><h3 class="text-base font-bold">Projects by Status</h3><div class="relative h-40 w-40 mx-auto"><canvas id="statusChart"></canvas></div></div>
                    </div>
                    <div class="col-span-3"><h3 class="text-base font-bold text-center">Projects by Location</h3><div class="h-48"><canvas id="locationChart"></canvas></div></div>
                </div>
                 <!-- Charts Row 2 -->
                <div class="grid grid-cols-5 gap-6 h-52">
                     <div class="col-span-3"><h3 class="text-base font-bold text-center">Sector Status Breakdown</h3><div class="h-48"><canvas id="pillarStatusChart"></canvas></div></div>
                     <div class="col-span-2"><h3 class="text-base font-bold text-center">Projects by Sector</h3><div class="relative h-40 w-40 mx-auto"><canvas id="sectorChart"></canvas></div></div>
                </div>
                 <!-- Charts Row 3 -->
                <div class="mt-6"><h3 class="text-base font-bold text-center">Financial Burn-Down</h3><div class="h-48"><canvas id="financialChart"></canvas></div></div>
            </div>
        </div>
    </main>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- Form population logic ---
    const reportLevelSelect = document.getElementById('report_level');
    const locationSelect = document.getElementById('location_id');
    const wards = <?php echo json_encode($wards); ?>;
    const constituencies = <?php echo json_encode($constituencies); ?>;
    const selectedLevel = '<?php echo $_POST['report_level'] ?? 'county'; ?>';
    const selectedLocation = '<?php echo $_POST['location_id'] ?? ''; ?>';

    function updateLocationOptions() {
        const level = reportLevelSelect.value;
        locationSelect.innerHTML = '';
        if (level === 'county') {
            locationSelect.disabled = true; locationSelect.classList.add('bg-gray-100');
            const opt = document.createElement('option'); opt.value = 'all'; opt.textContent = 'All of Homa Bay County'; locationSelect.appendChild(opt);
        } else {
            locationSelect.disabled = false; locationSelect.classList.remove('bg-gray-100');
            const placeholder = document.createElement('option'); placeholder.value = ''; placeholder.textContent = `-- Select a ${level} --`; locationSelect.appendChild(placeholder);
            const options = level === 'ward' ? wards : constituencies;
            options.forEach(item => {
                const opt = document.createElement('option'); opt.value = item.id; opt.textContent = item.name;
                if(item.id == selectedLocation) opt.selected = true;
                locationSelect.appendChild(opt);
            });
        }
    }
    reportLevelSelect.addEventListener('change', updateLocationOptions);
    updateLocationOptions();

    // --- Chart generation logic ---
    const reportData = <?php echo $report_data_json ?? 'null'; ?>;
    const downloadBtn = document.getElementById('download-btn');
    const reportPreview = document.getElementById('report-preview');
    let charts = []; 

    function renderReport(data) {
        if (!data) return;
        charts.forEach(chart => chart.destroy()); charts = [];

        // Populate KPIs
        const kpiContainer = document.getElementById('kpi-container');
        kpiContainer.innerHTML = `
            <div class="bg-gray-100 p-3 rounded-lg"><p class="text-sm font-bold text-gray-500">TOTAL PROJECTS</p><p class="text-4xl font-extrabold text-gray-900">${data.kpis.total_projects || 0}</p></div>
            <div class="bg-gray-100 p-3 rounded-lg"><p class="text-sm font-bold text-gray-500">ONGOING</p><p class="text-4xl font-extrabold text-brand-blue">${data.kpis.ongoing_projects || 0}</p></div>
            <div class="bg-gray-100 p-3 rounded-lg"><p class="text-sm font-bold text-gray-500">COMPLETED</p><p class="text-4xl font-extrabold text-green-600">${data.kpis.completed_projects || 0}</p></div>
            <div class="bg-gray-100 p-3 rounded-lg"><p class="text-sm font-bold text-gray-500">INVESTMENT</p><p class="text-2xl font-extrabold">KES ${((data.kpis.total_investment || 0)/1000000).toFixed(1)}M</p></div>
        `;

        const chartColors = ['#1E3A8A', '#F97316', '#6D28D9', '#16A34A', '#EAB308', '#DC2626', '#4B5563'];
        const statusColors = {'Completed': '#16A34A', 'Ongoing': '#1E3A8A', 'Planned': '#EAB308', 'Stalled': '#DC2626'};
        const commonOptions = { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } };

        charts.push(new Chart(document.getElementById('statusChart'), { type: 'doughnut', data: { labels: Object.keys(data.status), datasets: [{ data: Object.values(data.status), backgroundColor: chartColors, borderWidth: 0 }] }, options: { ...commonOptions, plugins: { legend: { position: 'right', labels: { boxWidth: 10, font: {size: 10} } } } } }));
        charts.push(new Chart(document.getElementById('sectorChart'), { type: 'pie', data: { labels: Object.keys(data.sector), datasets: [{ data: Object.values(data.sector), backgroundColor: chartColors, borderWidth: 0 }] }, options: { ...commonOptions, plugins: { legend: { position: 'bottom', labels: { boxWidth: 10, font: {size: 9} } } } }}));
        charts.push(new Chart(document.getElementById('locationChart'), { type: 'bar', data: { labels: Object.keys(data.location), datasets: [{ label: 'Projects', data: Object.values(data.location), backgroundColor: '#1E3A8A' }] }, options: { ...commonOptions, indexAxis: 'y', scales: { y: { ticks: { font: { size: 10 } } } } } }));
        
        const financialData = data.financial_burn.reduce((acc, item) => {
            acc.labels.push(item.month);
            acc.allocated.push((acc.allocated.slice(-1)[0] || 0) + parseFloat(item.allocated));
            acc.spent.push((acc.spent.slice(-1)[0] || 0) + parseFloat(item.spent));
            return acc;
        }, { labels: [], allocated: [], spent: [] });
        charts.push(new Chart(document.getElementById('financialChart'), { type: 'line', data: { labels: financialData.labels, datasets: [ { label: 'Budget Allocated', data: financialData.allocated, borderColor: '#6D28D9', fill: false, tension: 0.1 }, { label: 'Budget Spent', data: financialData.spent, borderColor: '#F97316', fill: false, tension: 0.1 } ] }, options: { ...commonOptions, plugins: { legend: { position: 'bottom' } } } }));

        const pillarStatusData = data.pillar_status.reduce((acc, item) => {
            if (!acc.labels.includes(item.title)) acc.labels.push(item.title);
            if (!acc.datasets[item.status]) {
                acc.datasets[item.status] = { label: item.status, data: Array(acc.labels.length).fill(0), backgroundColor: statusColors[item.status] };
            }
            return acc;
        }, { labels: [], datasets: {} });
        data.pillar_status.forEach(item => {
            const index = pillarStatusData.labels.indexOf(item.title);
            pillarStatusData.datasets[item.status].data[index] = item.count;
        });
        charts.push(new Chart(document.getElementById('pillarStatusChart'), { type: 'bar', data: { labels: pillarStatusData.labels, datasets: Object.values(pillarStatusData.datasets) }, options: { ...commonOptions, scales: { x: { stacked: true }, y: { stacked: true, ticks: {font: {size: 9}} } }, plugins: { legend: { position: 'bottom', labels:{ boxWidth: 10, font: {size: 9} } } } } }));

        downloadBtn.disabled = false;
    }
    
    if(reportData) {
        renderReport(reportData);
    }
    
    // Download logic
    downloadBtn.addEventListener('click', function() {
        downloadBtn.textContent = 'Generating...'; downloadBtn.disabled = true;
        html2canvas(reportPreview, { scale: 3, useCORS: true }).then(canvas => {
            const link = document.createElement('a');
            link.download = 'Genowa_Visual_Report.png';
            link.href = canvas.toDataURL('image/png');
            link.click();
            downloadBtn.textContent = 'Download as PNG'; downloadBtn.disabled = false;
        });
    });
});
</script>
<?php include 'includes/footer.php'; ?>
