<?php
// FILE: admin/user_management.php
// PURPOSE: A page for SuperAdmins to approve, assign roles to, and manage users.

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- DEPENDENCIES & HELPERS ---
require_once __DIR__ . '/vendor/autoload.php'; // Load Composer dependencies (like PHPMailer)
require_once 'config.php'; // Loads configuration from config.ini
require_once 'functions.php';
require_once 'db.php';
require_once 'includes/mailer_template.php'; // Loads the reusable email sending function

// Security: Only SuperAdmins can access this page
require_login();
require_super_admin();

$feedback_message = '';
$feedback_type = ''; // 'success' or 'error'

// --- GENERATE CSRF TOKEN ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];


// --- Handle POST Actions (Approve, Deactivate, Update Role) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // CSRF Validation
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $feedback_message = "Invalid or expired request. Please try again.";
        $feedback_type = 'error';
        error_log("CSRF Token Mismatch in User Management from user ID: {$_SESSION['id']}");
    } else {
        $user_id_to_action = $_POST['user_id'] ?? null;
        $current_super_admin_id = $_SESSION['id'];
        $action = $_POST['action'] ?? '';

        if ($user_id_to_action) {
            try {
                switch ($action) {
                    case 'update_role':
                        $new_role = $_POST['role'] ?? '';
                        if ($user_id_to_action == $current_super_admin_id) {
                            $feedback_message = "For security, you cannot change your own role.";
                            $feedback_type = 'error';
                        } elseif (in_array($new_role, ['Inputer', 'Approver', 'Admin', 'SuperAdmin'])) {
                            $stmt = $pdo->prepare("UPDATE users SET role = :role WHERE id = :id");
                            $stmt->execute(['role' => $new_role, 'id' => $user_id_to_action]);
                            $feedback_message = "User role updated successfully.";
                            $feedback_type = 'success';
                            error_log("Admin ID {$current_super_admin_id} updated role for user ID {$user_id_to_action} to {$new_role}");
                        } else {
                            $feedback_message = "Invalid role selected.";
                            $feedback_type = 'error';
                        }
                        break;

                    case 'approve_user':
                        $stmt = $pdo->prepare("UPDATE users SET is_active = 1 WHERE id = :id");
                        $stmt->execute(['id' => $user_id_to_action]);
                        $feedback_message = 'User has been approved and is now active.';
                        $feedback_type = 'success';
                        error_log("Admin ID {$current_super_admin_id} approved user ID {$user_id_to_action}");

                        // --- SEND APPROVAL EMAIL ---
                        try {
                            // Fetch user details for the email
                            $user_stmt = $pdo->prepare("SELECT name, email FROM users WHERE id = :id");
                            $user_stmt->execute(['id' => $user_id_to_action]);
                            $approved_user = $user_stmt->fetch(PDO::FETCH_ASSOC);

                            if ($approved_user) {
                                $subject = "Your Genowa Delivery Unit Account has been Approved";
                                
                                // Load and populate the HTML email template
                                $template_path = __DIR__ . '/includes/email_approval_template.html';
                                if(file_exists($template_path)) {
                                    $email_body = file_get_contents($template_path);
                                    $email_body = str_replace('{{user_name}}', e($approved_user['name']), $email_body);
                                    $email_body = str_replace('{{login_link}}', 'https://genowa.co.ke/admin/login.php', $email_body);

                                    $alt_body = "Hello " . e($approved_user['name']) . ",\n\nYour account on the Genowa Delivery Unit portal has been approved. You can now log in using your credentials at: https://genowa.co.ke/admin/login.php\n\nRegards,\nThe Genowa Team";

                                    // FIX: Call the renamed function to avoid conflicts
                                    if (send_styled_email($approved_user['email'], $approved_user['name'], $subject, $email_body, $alt_body, $config)) {
                                        $feedback_message .= ' An approval email has been sent to the user.';
                                    } else {
                                        $feedback_message .= ' (But could not send approval email.)';
                                        $feedback_type = 'error'; // Downgrade feedback type if email fails
                                    }
                                } else {
                                     error_log("Failed to send approval email: Template file not found at " . $template_path);
                                     $feedback_message .= ' (Could not send approval email: template missing.)';
                                     $feedback_type = 'error';
                                }
                            }
                        } catch (Exception $e) {
                            error_log("Failed to send approval email for user ID {$user_id_to_action}: " . $e->getMessage());
                            $feedback_message .= ' (Could not send approval email due to an error.)';
                            $feedback_type = 'error';
                        }
                        break;
                    
                    case 'deactivate_user':
                        if ($user_id_to_action == $current_super_admin_id) {
                            $feedback_message = "You cannot deactivate your own account.";
                            $feedback_type = 'error';
                        } else {
                            $stmt = $pdo->prepare("UPDATE users SET is_active = 0, deleted_at = NOW() WHERE id = :id");
                            $stmt->execute(['id' => $user_id_to_action]);
                            $feedback_message = 'User has been deactivated successfully.';
                            $feedback_type = 'success';
                            error_log("Admin ID {$current_super_admin_id} deactivated user ID {$user_id_to_action}");
                        }
                        break;
                    
                    case 'reactivate_user':
                        $stmt = $pdo->prepare("UPDATE users SET is_active = 1, deleted_at = NULL WHERE id = :id");
                        $stmt->execute(['id' => $user_id_to_action]);
                        $feedback_message = 'User has been reactivated successfully.';
                        $feedback_type = 'success';
                        error_log("Admin ID {$current_super_admin_id} reactivated user ID {$user_id_to_action}");
                        break;
                }
            } catch (PDOException $e) {
                $feedback_message = 'A database error occurred. The action could not be completed.';
                $feedback_type = 'error';
                error_log("User Management PDOException: " . $e->getMessage());
            }
        }
    }
}

// --- Fetch All Users for Display ---
try {
    $stmt = $pdo->query("SELECT u.id, u.name, u.email, u.phone_number, u.role, u.is_active, u.created_at, u.last_login_at, u.deleted_at, w.name as ward_name 
                         FROM users u 
                         LEFT JOIN wards w ON u.ward_id = w.id 
                         ORDER BY u.created_at DESC");
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $db_error = "Error fetching users: " . $e->getMessage();
}

$roles = ['Inputer', 'Approver', 'Admin', 'SuperAdmin'];

include 'includes/header.php';
?>

<style>
    :root {
        --brand-blue: #1e40af;
        --brand-orange: #f97316;
        --brand-purple: #7e22ce;
    }
    .focus-ring {
        --tw-ring-color: var(--brand-purple);
    }
</style>

<div class="min-h-full bg-gray-50">
    <header class="bg-white shadow-sm">
        <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8 flex justify-between items-center">
            <h1 class="text-3xl font-bold tracking-tight text-gray-900">User Management</h1>
        </div>
    </header>

    <main>
        <div class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
            <?php if ($feedback_message): ?>
                <div class="mb-6 p-4 rounded-lg <?php echo $feedback_type === 'success' ? 'bg-green-100 border border-green-300 text-green-800' : 'bg-red-100 border border-red-300 text-red-800'; ?>">
                    <p><?php echo e($feedback_message); ?></p>
                </div>
            <?php endif; ?>

            <div class="bg-white p-6 rounded-xl shadow-md mb-8">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div class="relative">
                        <label for="search-user" class="block text-sm font-medium text-gray-700 mb-1">Search by Name/Email</label>
                        <div class="pointer-events-none absolute inset-y-0 left-0 top-7 flex items-center pl-3">
                            <svg class="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor"><path fill-rule="evenodd" d="M9 3.5a5.5 5.5 0 100 11 5.5 5.5 0 000-11zM2 9a7 7 0 1112.452 4.391l3.328 3.329a.75.75 0 11-1.06 1.06l-3.329-3.328A7 7 0 012 9z" clip-rule="evenodd" /></svg>
                        </div>
                        <input type="text" id="search-user" class="block w-full rounded-lg border-gray-300 py-2.5 pl-10 shadow-sm focus:border-brand-purple focus:ring-brand-purple sm:text-sm focus-ring" placeholder="e.g. John Doe">
                    </div>
                    <div>
                        <label for="filter-role" class="block text-sm font-medium text-gray-700 mb-1">Filter by Role</label>
                        <select id="filter-role" class="block w-full rounded-lg border-gray-300 py-2.5 shadow-sm focus:border-brand-purple focus:ring-brand-purple sm:text-sm focus-ring">
                            <option value="">All Roles</option>
                            <?php foreach ($roles as $role): ?>
                                <option value="<?php echo e($role); ?>"><?php echo e($role); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="filter-status" class="block text-sm font-medium text-gray-700 mb-1">Filter by Status</label>
                        <select id="filter-status" class="block w-full rounded-lg border-gray-300 py-2.5 shadow-sm focus:border-brand-purple focus:ring-brand-purple sm:text-sm focus-ring">
                            <option value="">All Statuses</option>
                            <option value="active">Active</option>
                            <option value="pending">Pending</option>
                            <option value="deactivated">Deactivated</option>
                        </select>
                    </div>
                </div>
            </div>

             <div class="overflow-hidden bg-white rounded-xl shadow-md">
                 <div class="overflow-x-auto">
                     <?php if (isset($db_error)): ?>
                         <div class="p-6 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
                     <?php else: ?>
                         <table class="min-w-full">
                             <thead class="bg-gray-50 border-b border-gray-200">
                                 <tr>
                                     <th scope="col" class="px-6 py-4 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">User</th>
                                     <th scope="col" class="px-6 py-4 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">Role</th>
                                     <th scope="col" class="px-6 py-4 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">Status</th>
                                     <th scope="col" class="px-6 py-4 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">Timestamps</th>
                                     <th scope="col" class="px-6 py-4 text-left text-sm font-semibold text-gray-600 uppercase tracking-wider">Actions</th>
                                 </tr>
                             </thead>
                             <tbody id="user-table-body" class="divide-y divide-gray-200">
                                 <?php foreach ($users as $user): ?>
                                     <tr class="hover:bg-gray-50 transition-colors duration-200" data-name="<?php echo e(strtolower($user['name'])); ?>" data-email="<?php echo e(strtolower($user['email'])); ?>" data-role="<?php echo e($user['role']); ?>" data-status="<?php echo $user['deleted_at'] ? 'deactivated' : ($user['is_active'] ? 'active' : 'pending'); ?>">
                                         <td class="px-6 py-4 whitespace-nowrap">
                                             <div class="text-base font-semibold text-gray-800"><?php echo e($user['name']); ?></div>
                                             <div class="text-sm text-gray-600"><?php echo e($user['email']); ?></div>
                                             <div class="text-sm text-gray-500"><?php echo e($user['phone_number'] ?? 'N/A'); ?></div>
                                         </td>
                                         <td class="px-6 py-4 whitespace-nowrap">
                                             <form method="POST" action="user_management.php">
                                                 <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                                                 <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                                 <input type="hidden" name="action" value="update_role">
                                                 <select name="role" class="block w-full rounded-lg border-gray-300 py-2 shadow-sm focus:border-brand-purple focus:ring-brand-purple sm:text-sm focus-ring disabled:bg-gray-100 disabled:cursor-not-allowed" onchange="this.form.submit()" <?php if ($user['id'] == $_SESSION['id']) echo 'disabled'; ?>>
                                                     <?php foreach ($roles as $role): ?>
                                                         <option value="<?php echo $role; ?>" <?php if ($user['role'] == $role) echo 'selected'; ?>><?php echo $role; ?></option>
                                                     <?php endforeach; ?>
                                                 </select>
                                             </form>
                                         </td>
                                         <td class="px-6 py-4 whitespace-nowrap">
                                             <?php if ($user['deleted_at']): ?>
                                                 <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-gray-200 text-gray-800">Deactivated</span>
                                             <?php elseif ($user['is_active']): ?>
                                                 <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-green-100 text-green-800">Active</span>
                                             <?php else: ?>
                                                 <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">Pending</span>
                                             <?php endif; ?>
                                         </td>
                                          <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
                                             <div><span class="font-semibold">Registered:</span> <?php echo date('M j, Y', strtotime($user['created_at'])); ?></div>
                                             <div><span class="font-semibold">Last Login:</span> <?php echo $user['last_login_at'] ? date('M j, Y H:i', strtotime($user['last_login_at'])) : 'Never'; ?></div>
                                         </td>
                                         <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                             <div class="flex items-center space-x-3">
                                                 <form method="POST" action="user_management.php">
                                                     <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                                                     <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                                     
                                                     <?php if ($user['deleted_at']): ?>
                                                         <button type="submit" name="action" value="reactivate_user" class="px-3 py-1.5 text-sm rounded-md font-semibold text-white bg-blue-600 hover:bg-blue-700 transition-colors duration-200">Reactivate</button>
                                                     <?php else: ?>
                                                         <?php if (!$user['is_active']): ?>
                                                             <button type="submit" name="action" value="approve_user" class="px-3 py-1.5 text-sm rounded-md font-semibold text-white bg-green-600 hover:bg-green-700 transition-colors duration-200">Approve</button>
                                                         <?php endif; ?>
                                                     <?php endif; ?>
                                                 </form>
                                                 <?php if (!$user['deleted_at'] && $user['id'] != $_SESSION['id']): ?>
                                                     <button type="button" data-action="deactivate" data-user-name="<?php echo e($user['name']); ?>" data-form-id="deactivate-form-<?php echo e($user['id']); ?>" class="px-3 py-1.5 text-sm rounded-md font-semibold text-white bg-red-600 hover:bg-red-700 transition-colors duration-200 modal-trigger">Deactivate</button>
                                                 <?php endif; ?>
                                             </div>
                                             <form id="deactivate-form-<?php echo e($user['id']); ?>" method="POST" action="user_management.php" class="hidden">
                                                 <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                                                 <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                                 <input type="hidden" name="action" value="deactivate_user">
                                             </form>
                                         </td>
                                     </tr>
                                 <?php endforeach; ?>
                             </tbody>
                         </table>
                     <?php endif; ?>
                 </div>
             </div>
        </div>
    </main>
</div>

<div id="confirmation-modal" class="fixed z-50 inset-0 overflow-y-auto hidden" aria-labelledby="modal-title" role="dialog" aria-modal="true">
  <div class="flex items-end justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
    <div class="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" aria-hidden="true"></div>
    <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
    <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
      <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
        <div class="sm:flex sm:items-start">
          <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-100 sm:mx-0 sm:h-10 sm:w-10">
            <svg class="h-6 w-6 text-red-600" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" aria-hidden="true">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
            </svg>
          </div>
          <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left">
            <h3 class="text-lg leading-6 font-medium text-gray-900" id="modal-title">Deactivate Account</h3>
            <div class="mt-2">
              <p class="text-sm text-gray-600" id="modal-body">Are you sure you want to deactivate this account? This will prevent the user from logging in but their data will be preserved.</p>
            </div>
          </div>
        </div>
      </div>
      <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
        <button type="button" id="confirm-button" class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-red-600 text-base font-medium text-white hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 sm:ml-3 sm:w-auto sm:text-sm">
          Deactivate
        </button>
        <button type="button" id="cancel-button" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
          Cancel
        </button>
      </div>
    </div>
  </div>
</div>


<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- Live Search and Filtering Logic ---
    const searchInput = document.getElementById('search-user');
    const roleFilter = document.getElementById('filter-role');
    const statusFilter = document.getElementById('filter-status');
    const userTableBody = document.getElementById('user-table-body');
    const tableRows = userTableBody.getElementsByTagName('tr');

    function filterUsers() {
        const searchTerm = searchInput.value.toLowerCase();
        const roleValue = roleFilter.value;
        const statusValue = statusFilter.value;

        for (let i = 0; i < tableRows.length; i++) {
            const row = tableRows[i];
            const name = row.dataset.name;
            const email = row.dataset.email;
            const role = row.dataset.role;
            const status = row.dataset.status;

            const matchesSearch = name.includes(searchTerm) || email.includes(searchTerm);
            const matchesRole = roleValue === '' || role === roleValue;
            const matchesStatus = statusValue === '' || status === statusValue;

            if (matchesSearch && matchesRole && matchesStatus) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        }
    }

    searchInput.addEventListener('keyup', filterUsers);
    roleFilter.addEventListener('change', filterUsers);
    statusFilter.addEventListener('change', filterUsers);

    // --- Modal Logic ---
    const modal = document.getElementById('confirmation-modal');
    const modalTitle = document.getElementById('modal-title');
    const modalBody = document.getElementById('modal-body');
    const confirmButton = document.getElementById('confirm-button');
    const cancelButton = document.getElementById('cancel-button');
    let formToSubmit = null;

    document.querySelectorAll('.modal-trigger').forEach(button => {
        button.addEventListener('click', function() {
            const userName = this.dataset.userName;
            const action = this.dataset.action;
            formToSubmit = document.getElementById(this.dataset.formId);

            modalTitle.textContent = `Deactivate "${userName}"`;
            modalBody.textContent = `Are you sure you want to deactivate this user? They will no longer be able to log in, but their data will be preserved. This action can be undone.`;
            confirmButton.textContent = 'Deactivate';
            modal.classList.remove('hidden');
        });
    });

    confirmButton.addEventListener('click', function() {
        if (formToSubmit) {
            formToSubmit.submit();
        }
    });

    function closeModal() {
        modal.classList.add('hidden');
        formToSubmit = null;
    }

    cancelButton.addEventListener('click', closeModal);
    modal.addEventListener('click', function(event) {
        if (event.target === modal) {
            closeModal();
        }
    });
});
</script>

<?php include 'includes/footer.php'; ?>
