<?php
// FILE: admin/project_delete.php
// PURPOSE: Handles the deletion of a project and its associated photos.

require_once 'functions.php';
require_once 'db.php';

// Ensure user is logged in
require_login();

// Check if an ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("location: dashboard.php");
    exit;
}

$project_id = $_GET['id'];

try {
    $pdo->beginTransaction(); // Start a transaction

    // --- Security Check & Fetch Photo Files ---
    // First, get the project details to verify ownership and get photo filenames.
    $check_sql = "SELECT ward_id FROM projects WHERE id = :id";
    $stmt = $pdo->prepare($check_sql);
    $stmt->execute(['id' => $project_id]);
    $project = $stmt->fetch();

    if (!$project) {
        // Project doesn't exist
        header("location: dashboard.php");
        exit;
    }

    // Security: WardAdmins can only delete projects in their own ward.
    if (!is_super_admin() && $project['ward_id'] != $_SESSION['ward_id']) {
        // Unauthorized attempt, redirect away.
        header("location: dashboard.php");
        exit;
    }
    
    // Fetch all associated photo filenames BEFORE deleting the project record
    $photos_stmt = $pdo->prepare("SELECT image_url FROM project_photos WHERE project_id = :project_id");
    $photos_stmt->execute(['project_id' => $project_id]);
    $photos = $photos_stmt->fetchAll(PDO::FETCH_COLUMN);

    // --- Delete from Database ---
    // The ON DELETE CASCADE constraint will automatically delete rows from `project_photos`.
    $delete_stmt = $pdo->prepare("DELETE FROM projects WHERE id = :id");
    $delete_stmt->execute(['id' => $project_id]);

    // --- Delete Files from Server ---
    $upload_dir = 'uploads/';
    foreach ($photos as $filename) {
        $file_path = $upload_dir . $filename;
        if (file_exists($file_path)) {
            @unlink($file_path); // Use @ to suppress errors if file doesn't exist for some reason
        }
    }

    $pdo->commit(); // Commit the transaction

} catch (PDOException $e) {
    $pdo->rollBack(); // Rollback on error
    // In a real application, you would log this error.
    // For now, we'll just redirect with an error message (optional).
    // For simplicity, we redirect to the dashboard.
    header("location: dashboard.php?error=deletion_failed");
    exit;
}

// Redirect back to the dashboard on successful deletion
header("location: dashboard.php?success=project_deleted");
exit;
