<?php
// FILE: admin/login.php
// PURPOSE: Enterprise-grade login page with brute-force protection, enhanced logging, and improved UX.

// --- CONFIGURATION ---
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOGIN_ATTEMPT_WINDOW', 15); // In minutes

// No need to start session here, it's handled later.
$error_message = '';
$is_ip_blocked = false;

require_once 'db.php'; // DB connection is needed earlier for IP check.

// --- BRUTE-FORCE PROTECTION: IP CHECK ---
try {
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $stmt = $pdo->prepare(
        "SELECT COUNT(*) FROM login_attempts WHERE ip_address = :ip_address AND attempt_time > (NOW() - INTERVAL :window MINUTE)"
    );
    $stmt->execute(['ip_address' => $ip_address, 'window' => LOGIN_ATTEMPT_WINDOW]);
    $attempts_count = $stmt->fetchColumn();

    if ($attempts_count >= MAX_LOGIN_ATTEMPTS) {
        $is_ip_blocked = true;
        $error_message = "You have made too many failed login attempts. Please try again in " . LOGIN_ATTEMPT_WINDOW . " minutes.";
    }
} catch (PDOException $e) {
    error_log("IP Check Failed: " . $e->getMessage());
    $error_message = "A security check failed. Please try again later.";
    $is_ip_blocked = true; // Block further action if the check fails
}


// --- FORM SUBMISSION LOGIC ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && !$is_ip_blocked) {
    
    $email = trim($_POST['email']);
    $password = trim($_POST['password']);
    $remember_me = isset($_POST['remember-me']);

    if (empty($email) || empty($password)) {
        $error_message = "Please enter both email and password.";
    } else {
        try {
            $stmt = $pdo->prepare("SELECT id, name, email, password, role, ward_id, is_active FROM users WHERE email = :email LIMIT 1");
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch();

            if ($user && password_verify($password, $user['password'])) {
                // --- LOGIN SUCCESS ---
                
                // Clear any previous failed login attempts for this IP
                $clear_stmt = $pdo->prepare("DELETE FROM login_attempts WHERE ip_address = :ip_address");
                $clear_stmt->execute(['ip_address' => $ip_address]);

                if ($user['is_active'] != 1) {
                    $error_message = "Your account is pending approval. Please contact an administrator.";
                    error_log("Login Failure: Account pending approval for user '$email' from IP: $ip_address");
                } else {
                    // Account is valid. Start session.
                    if (session_status() == PHP_SESSION_NONE) {
                        session_start();
                    }
                    session_regenerate_id(true);

                    $_SESSION["loggedin"] = true;
                    $_SESSION["id"] = $user['id'];
                    $_SESSION["name"] = $user['name'];
                    $_SESSION["role"] = $user['role'];
                    $_SESSION["ward_id"] = $user['ward_id'];
                    
                    // --- ENTERPRISE HOOK: TWO-FACTOR AUTHENTICATION (2FA) ---
                    // At this point, you would check if the user has 2FA enabled.
                    // If so, you would NOT redirect to the dashboard yet.
                    // Instead, you'd set a temporary session variable, e.g., $_SESSION['2fa_pending'] = true;
                    // and redirect to a page asking for their 2FA code.
                    // The dashboard would then check for $_SESSION["loggedin"] AND !isset($_SESSION['2fa_pending']).

                    error_log("Login Success: User '{$user['name']}' logged in from IP: $ip_address");

                    if ($remember_me) {
                        $selector = bin2hex(random_bytes(16));
                        $validator = bin2hex(random_bytes(32));
                        $token = $selector . ':' . $validator;
                        $hashed_validator = password_hash($validator, PASSWORD_DEFAULT);
                        $expires = time() + (86400 * 30);
                        
                        $token_stmt = $pdo->prepare(
                            "INSERT INTO auth_tokens (selector, hashed_validator, user_id, expires) VALUES (:selector, :hashed_validator, :user_id, :expires)"
                        );
                        $token_stmt->execute([
                            'selector' => $selector,
                            'hashed_validator' => $hashed_validator,
                            'user_id' => $user['id'],
                            'expires' => date('Y-m-d H:i:s', $expires)
                        ]);

                        setcookie('remember_me', $token, $expires, '/', '', false, true);
                    }

                    header("location: dashboard.php");
                    exit;
                }

            } else {
                // --- LOGIN FAILURE ---
                $error_message = "The email or password you entered is incorrect.";
                
                // Record the failed attempt
                $fail_stmt = $pdo->prepare("INSERT INTO login_attempts (ip_address) VALUES (:ip_address)");
                $fail_stmt->execute(['ip_address' => $ip_address]);
                
                // Log the failure for security auditing
                error_log("Login Failure: Invalid credentials for user '$email' from IP: $ip_address");
            }
        } catch (PDOException $e) {
            $error_message = "A database error occurred. Please try again later.";
            error_log("Login PDOException: " . $e->getMessage()); 
        }
    }
}

// Now include the header, as we are done with any potential redirects.
include 'includes/header.php';
?>

<!-- The rest of your HTML form remains exactly the same -->
<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        
        <div>
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Admin Panel Sign In
            </h2>
        </div>

        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if (!empty($error_message)): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                    <p><?php echo htmlspecialchars($error_message); ?></p>
                </div>
            <?php endif; ?>
    
            <form id="login-form" class="space-y-6" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
                <div>
                    <label for="email-address" class="sr-only">Email address</label>
                    <input id="email-address" name="email" type="email" autocomplete="email" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Email address" <?php if($is_ip_blocked) echo 'disabled'; ?>>
                </div>
                <div>
                    <label for="password" class="sr-only">Password</label>
                    <input id="password" name="password" type="password" autocomplete="current-password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Password" <?php if($is_ip_blocked) echo 'disabled'; ?>>
                </div>
    
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <input id="remember-me" name="remember-me" type="checkbox" class="h-4 w-4 text-brand-orange focus:ring-brand-purple border-gray-300 rounded" <?php if($is_ip_blocked) echo 'disabled'; ?>>
                        <label for="remember-me" class="ml-2 block text-sm text-gray-900">
                            Remember me
                        </label>
                    </div>
    
                    <div class="text-sm">
                        <a href="forgot_password.php" class="font-medium text-brand-purple hover:text-brand-orange">
                            Forgot your password?
                        </a>
                    </div>
                </div>
    
                <div>
                    <button id="signin-button" type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out disabled:bg-orange-300 disabled:cursor-not-allowed" <?php if($is_ip_blocked) echo 'disabled'; ?>>
                        <span class="absolute left-0 inset-y-0 flex items-center pl-3">
                            <!-- Heroicon name: solid/lock-closed -->
                            <svg class="h-5 w-5 text-orange-400 group-hover:text-orange-300" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                <path fill-rule="evenodd" d="M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z" clip-rule="evenodd" />
                            </svg>
                        </span>
                        <span class="button-text">Sign in</span>
                        <!-- NEW: Loading Spinner -->
                        <div class="loader ease-linear rounded-full border-4 border-t-4 border-gray-200 h-5 w-5 ml-3 hidden"></div>
                    </button>
                </div>
            </form>
            <p class="text-center text-sm text-gray-600">
                Don't have an account? 
                <a href="signup.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign up here
                </a>
            </p>
        </div>
    </div>
</div>

<!-- NEW: JAVASCRIPT FOR UX IMPROVEMENTS -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const loginForm = document.getElementById('login-form');
    const signinButton = document.getElementById('signin-button');
    const buttonText = signinButton.querySelector('.button-text');
    const loader = signinButton.querySelector('.loader');

    if (loginForm) {
        loginForm.addEventListener('submit', function(e) {
            // Basic validation check
            const email = document.getElementById('email-address').value;
            const password = document.getElementById('password').value;

            if (email.trim() === '' || password.trim() === '') {
                // Let the browser handle the 'required' attribute validation
                return;
            }

            // Disable button and show loader
            signinButton.disabled = true;
            buttonText.classList.add('hidden');
            loader.classList.remove('hidden');
        });
    }
});
</script>

<style>
/* Simple CSS for the loader animation */
.loader {
    border-top-color: #f97316; /* brand-orange */
    -webkit-animation: spinner 1.5s linear infinite;
    animation: spinner 1.5s linear infinite;
}

@-webkit-keyframes spinner {
    0% { -webkit-transform: rotate(0deg); }
    100% { -webkit-transform: rotate(360deg); }
}

@keyframes spinner {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}
</style>

<?php
// Include the footer template
include 'includes/footer.php';
?>
