<?php
// FILE: admin/includes/mailer_template.php
// PURPOSE: Provides a reusable function for sending emails via PHPMailer using settings from config.ini.

// FIX: Load the Composer autoloader from within this file to ensure PHPMailer classes are always available.
// The path goes up one level from 'includes' to the 'admin' directory to find the 'vendor' folder.
require_once __DIR__ . '/../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * Sends a styled HTML email using globally defined configuration.
 *
 * @param string $recipient_email The email address of the recipient.
 * @param string $recipient_name The name of the recipient.
 * @param string $subject The subject of the email.
 * @param string $html_body The HTML content of the email.
 * @param string $alt_body The plain text alternative content.
 * @param array $config The global configuration array containing SMTP settings.
 * @return bool True on success, false on failure.
 */
function send_styled_email($recipient_email, $recipient_name, $subject, $html_body, $alt_body, $config) {
    if (!isset($config['smtp'])) {
        error_log("Mailer Error: SMTP configuration is missing from config.ini.");
        return false;
    }
    
    $smtp_config = $config['smtp'];
    $mail = new PHPMailer(true);

    try {
        // Server settings
        // $mail->SMTPDebug = 2; // Enable verbose debug output for troubleshooting
        $mail->isSMTP();
        $mail->Host       = $smtp_config['host'];
        $mail->SMTPAuth   = true;
        $mail->Username   = $smtp_config['username'];
        $mail->Password   = $smtp_config['password'];
        $mail->SMTPSecure = $smtp_config['encryption'] === 'ssl' ? PHPMailer::ENCRYPTION_SMTPS : PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port       = (int)$smtp_config['port']; // Ensure port is an integer

        // Recipients
        $from_email = $smtp_config['from_email'] ?? $smtp_config['username'];
        $from_name = $smtp_config['from_name'] ?? 'Genowa Delivery Unit';
        $mail->setFrom($from_email, $from_name);
        
        $mail->addAddress($recipient_email, $recipient_name);

        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body    = $html_body;
        $mail->AltBody = $alt_body;

        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("PHPMailer Error: {$mail->ErrorInfo}");
        return false;
    }
}
