<?php
// FILE: admin/handle_approval.php
// PURPOSE: Securely processes project approval actions (Approve, Request Changes).
// VERSION: 2.0 - Added logic to require and save comments for "Request Changes" action.

require_once 'functions.php';
require_once 'db.php';
require_login();

// --- Permissions Check ---
if (!is_approver() && !is_super_admin()) {
    $_SESSION['flash_message'] = "Error: You do not have permission to perform this action.";
    header("location: dashboard.php");
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $project_id = filter_input(INPUT_POST, 'project_id', FILTER_VALIDATE_INT);
    $action = $_POST['action'] ?? '';
    $comment = trim($_POST['comment'] ?? '');

    if (!$project_id || !in_array($action, ['approve', 'request_changes'])) {
        $_SESSION['flash_message'] = "Error: Invalid request.";
        header("location: approvals.php");
        exit;
    }

    // --- Specific validation for 'request_changes' ---
    if ($action === 'request_changes' && empty($comment)) {
        $_SESSION['flash_message'] = "Error: A comment is required when requesting changes.";
        header("location: approvals.php");
        exit;
    }

    $new_status_code = null;
    $log_message = '';

    if ($action === 'approve') {
        $new_status_code = 1; // 1 = Approved
        $log_message = 'Approved Project';
        $_SESSION['flash_message'] = "Project has been successfully approved and is now public.";
    } elseif ($action === 'request_changes') {
        $new_status_code = 2; // 2 = Changes Requested
        $log_message = 'Changes Requested for Project';
        $_SESSION['flash_message'] = "Project status has been set to 'Changes Requested'.";
    }

    try {
        $pdo->beginTransaction();

        // 1. Update the project status
        if ($action === 'approve') {
             $sql = "UPDATE projects SET is_approved = :status, approved_by = :user_id, approved_at = NOW() WHERE id = :id";
             $stmt = $pdo->prepare($sql);
             $stmt->execute([
                'status' => $new_status_code,
                'user_id' => $_SESSION['id'],
                'id' => $project_id
             ]);
        } else {
            $sql = "UPDATE projects SET is_approved = :status WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'status' => $new_status_code,
                'id' => $project_id
            ]);
        }
        
        // 2. If changes are requested, save the comment
        if ($action === 'request_changes') {
            $comment_sql = "INSERT INTO project_approval_comments (project_id, user_id, comment, decision) VALUES (:project_id, :user_id, :comment, 'Changes Requested')";
            $comment_stmt = $pdo->prepare($comment_sql);
            $comment_stmt->execute([
                'project_id' => $project_id,
                'user_id' => $_SESSION['id'],
                'comment' => $comment
            ]);
        }
        
        // 3. Log this important action
        log_action($pdo, $log_message, 'Project', $project_id);
        
        $pdo->commit();

    } catch(PDOException $e) {
        $pdo->rollBack();
        $_SESSION['flash_message'] = "Database Error: Could not update the project status. " . $e->getMessage();
    }
}

header("location: approvals.php");
exit;
?>