<?php
// FILE: admin/generate_infographic.php
// PURPOSE: Dynamically generates a shareable infographic for a selected project.
// Supports three modes: Classic, Modern, and Project Information.

require_once 'functions.php';
require_once 'db.php';

require_login();

// --- Get Project ID from URL ---
$project_id = $_GET['id'] ?? null;
$project = null;
$all_photos = [];
$completion_percentage = 0;
$all_projects_list = [];

try {
    // Fetch list of all approved projects for the dropdown
    $projects_list_stmt = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC");
    $all_projects_list = $projects_list_stmt->fetchAll();

    // If an ID is provided, fetch the specific project data
    if ($project_id) {
        $sql = "SELECT p.*, w.name as ward_name, c.name as constituency_name
                FROM projects p
                JOIN wards w ON p.ward_id = w.id
                JOIN constituencies c ON w.constituency_id = c.id
                WHERE p.id = :id AND p.is_approved = 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['id' => $project_id]);
        $project = $stmt->fetch();

        if ($project) {
            $photos_stmt = $pdo->prepare("SELECT id, image_url, photo_type FROM project_photos WHERE project_id = :project_id ORDER BY photo_type, uploaded_at DESC");
            $photos_stmt->execute(['project_id' => $project_id]);
            $all_photos = $photos_stmt->fetchAll();

            if ($project['status'] === 'Completed') {
                $completion_percentage = 100;
            } elseif ($project['status'] === 'Ongoing' && $project['start_date'] && $project['expected_completion_date']) {
                $today = new DateTime();
                $start_date = new DateTime($project['start_date']);
                $expected_date = new DateTime($project['expected_completion_date']);
                if ($start_date <= $expected_date && $today >= $start_date) {
                    $total_duration = $start_date->diff($expected_date)->days;
                    $elapsed_duration = $start_date->diff($today)->days;
                    $completion_percentage = ($total_duration > 0) ? min(100, floor(($elapsed_duration / $total_duration) * 100)) : 100;
                }
            }
        }
    }
} catch (PDOException $e) {
    $db_error = "Database error: " . $e->getMessage();
}

include 'includes/header.php';
?>
<!-- Include libraries -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<link href="https://cdn.jsdelivr.net/npm/tom-select@2.2.2/dist/css/tom-select.bootstrap5.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/tom-select@2.2.2/dist/js/tom-select.complete.min.js"></script>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;800&display=swap" rel="stylesheet">


<style>
    /* Custom styles for TomSelect */
    .ts-control { padding: 0.75rem 1rem !important; border-radius: 0.5rem !important; }
    .text-shadow { text-shadow: 0 2px 8px rgba(0,0,0,0.7); }

    /* Infographic Container */
    #infographic-container {
        font-family: 'Poppins', sans-serif;
        width: 1080px;
        height: 1350px;
        position: relative;
        overflow: hidden;
        background-color: #1f2937; /* Fallback bg */
        transform-origin: top left;
    }

    /* Background Layers */
    .bg-layer {
        position: absolute;
        inset: 0;
        background-size: cover;
        background-position: center;
        transition: opacity 0.3s ease-in-out;
    }
    #template-layer { z-index: 1; }
    #photo-layer { z-index: 2; }
    #gradient-layer {
        z-index: 3;
        background: linear-gradient(to top, rgba(0,0,0,0.7), rgba(0,0,0,0.3), transparent);
    }

    /* Styling for the photo when it's inside a template */
    #photo-layer.photo-in-template {
        top: 180px;
        left: 50px;
        right: 50px;
        height: 1020px;
        bottom: auto;
        background-size: cover;
        background-repeat: no-repeat;
        border-radius: 32px;
        box-shadow: 0 10px 25px rgba(0,0,0,0.2);
    }

    /* Content wrapper for text and logos */
    .content-wrapper {
        position: relative;
        z-index: 4;
        height: 100%;
        display: flex;
        flex-direction: column;
        color: #FFFFFF; /* Default text color for classic mode */
    }

    /* Layout-specific visibility */
    .layout-banner-bottom .layout-content-overlay { display: none; }
    .layout-classic-overlay .layout-content-banner { display: none; }

    /* Mode-specific visibility rules */
    .mode-modern .logo-container,
    .mode-modern .status-badge,
    .mode-modern .hashtag-container,
    .mode-project-info .logo-container,
    .mode-project-info .status-badge,
    .mode-project-info .hashtag-container,
    .mode-project-info .infographic-caption {
        display: none !important;
    }

    /* In Modern/Project-Info modes, ensure text color is correctly applied */
    .mode-modern .text-area h3,
    .mode-modern .text-area .infographic-caption,
    .mode-modern .text-area .progress-bar-container p,
    .mode-project-info .text-area h3,
    .mode-project-info .text-area .progress-bar-container p {
        color: inherit; /* Inherits from content-wrapper which is set by JS */
    }
    
    /* Specific styling for Info Card to improve legibility */
    .mode-project-info .text-area {
        background-color: rgba(21, 30, 43, 0.85);
        padding: 2rem !important;
        border-radius: 1rem;
        backdrop-filter: blur(8px);
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.37) !important;
        text-shadow: none !important;
    }
    .mode-project-info .text-area p:first-child { /* Ward Name */
        font-size: 2.5rem !important; /* 40px */
    }
    .mode-project-info .text-area h3 { /* Project Name */
        font-size: 4.5rem !important; /* 72px */
        line-height: 1.1 !important;
    }


    /* Thumbnail selector */
    #thumbnail-selector img.selected {
        border-color: #5D3FD3; /* brand-purple */
        box-shadow: 0 0 0 3px #5D3FD3;
    }
    
    /* Controls visibility */
    .control-group { display: none; }
</style>


<header class="bg-white shadow">
    <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8">
        <h1 class="text-3xl font-bold tracking-tight text-gray-900">Generate Infographic</h1>
    </div>
</header>
<main class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
    
    <div class="bg-white p-6 rounded-lg shadow-lg mb-8">
        <form action="generate_infographic.php" method="get">
            <label for="project-select" class="block text-lg font-medium text-gray-900 mb-2">Select a Project</label>
            <div class="flex items-center space-x-2">
                <select id="project-select" name="id" placeholder="Search for a project..." class="flex-grow">
                    <option value="">Search for a project...</option>
                    <?php foreach ($all_projects_list as $p): ?>
                        <option value="<?php echo e($p['id']); ?>" <?php if($project_id == $p['id']) echo 'selected'; ?>><?php echo e($p['name']); ?></option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="rounded-md bg-brand-blue px-4 py-3 text-sm font-semibold text-white shadow-sm hover:bg-brand-purple">Load Project</button>
            </div>
        </form>
    </div>

    <?php if ($project): ?>
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
        
        <!-- ====== PREVIEW PANE ====== -->
        <div class="lg:col-span-1">
             <h2 class="text-xl font-bold mb-4">Live Preview</h2>
             <div id="preview-wrapper" class="w-full max-w-[540px] mx-auto aspect-[1080/1350]">
                <div id="infographic-container" class="layout-banner-bottom">
                    <!-- Background Layers -->
                    <div id="template-layer" class="bg-layer"></div>
                    <div id="photo-layer" class="bg-layer"></div>
                    <div id="gradient-layer" class="bg-layer"></div>

                    <!-- Content Wrapper -->
                    <div class="content-wrapper">
                        <!-- Layout: Banner Bottom -->
                        <div class="layout-content-banner flex-grow flex flex-col justify-between p-12">
                            <div class="flex justify-between items-start">
                                <div class="logo-container flex items-center space-x-4">
                                    <div class="hbc-logo-wrapper"><img src="assets/hbc_logo.png" alt="Homa Bay County Logo" class="h-16 w-auto" crossorigin="anonymous"></div>
                                    <div class="logo-divider w-px h-16 bg-white/50"></div>
                                    <div class="genowa-logo-wrapper"><p class="font-black text-4xl">GENOWA</p><p class="font-semibold text-xl text-brand-orange tracking-wider -mt-2">DELIVERY UNIT</p></div>
                                </div>
                                <div class="status-badge px-4 py-2 text-lg font-bold inline-block rounded-full bg-white/20 backdrop-blur-sm"><?php echo e($project['status']); ?></div>
                            </div>
                            <div class="text-area">
                                <div class="bg-brand-blue/90 p-6 rounded-lg backdrop-blur-sm shadow-lg border border-white/10">
                                    <p class="text-xl font-semibold text-brand-orange ward-name"><?php echo e($project['ward_name']); ?> Ward</p>
                                    <h3 class="text-4xl font-black leading-tight project-name"><?php echo e($project['name']); ?></h3>
                                    <?php if (($project['status'] === 'Ongoing' && $completion_percentage > 0) || $project['status'] === 'Completed'): ?>
                                    <div class="mt-4 progress-bar-container">
                                        <div class="w-full bg-white/20 rounded-full h-2.5"><div class="bg-brand-orange h-2.5 rounded-full" style="width: <?php echo $completion_percentage; ?>%"></div></div>
                                        <p class="text-right text-sm font-bold mt-1.5 text-gray-200"><?php echo $completion_percentage; ?>% <?php echo ($project['status'] === 'Completed') ? 'Complete' : 'Timeline Progress'; ?></p>
                                    </div>
                                    <?php endif; ?>
                                    <p class="infographic-caption mt-3 text-2xl text-gray-200 leading-relaxed"></p>
                                    <p class="hashtag-container text-right text-base font-bold mt-3 text-brand-orange/80">#GladysWangaDelivers</p>
                                </div>
                            </div>
                        </div>
                        <!-- Layout: Classic Overlay -->
                        <div class="layout-content-overlay flex-grow flex flex-col justify-between p-12">
                            <div class="flex justify-between items-start">
                                <div class="logo-container flex items-center space-x-4">
                                    <div class="hbc-logo-wrapper"><img src="assets/hbc_logo.png" alt="Homa Bay County Logo" class="h-16 w-auto text-shadow" crossorigin="anonymous"></div>
                                    <div class="logo-divider w-px h-16 bg-white/50"></div>
                                    <div class="genowa-logo-wrapper"><p class="font-black text-4xl text-shadow">GENOWA</p><p class="font-semibold text-xl text-brand-orange tracking-wider -mt-2 text-shadow">DELIVERY UNIT</p></div>
                                </div>
                                <div class="status-badge px-4 py-2 text-lg font-bold inline-block rounded-full bg-white/20 backdrop-blur-sm text-shadow"><?php echo e($project['status']); ?></div>
                            </div>
                             <div class="text-area text-shadow">
                                <p class="text-2xl font-semibold text-brand-orange ward-name"><?php echo e($project['ward_name']); ?> Ward</p>
                                <h3 class="text-5xl font-black leading-tight project-name"><?php echo e($project['name']); ?></h3>
                                <?php if (($project['status'] === 'Ongoing' && $completion_percentage > 0) || $project['status'] === 'Completed'): ?>
                                 <div class="mt-4 progress-bar-container"><div class="w-full bg-white/20 rounded-full h-2.5"><div class="bg-brand-orange h-2.5 rounded-full" style="width: <?php echo $completion_percentage; ?>%"></div></div><p class="text-right text-sm font-bold mt-1.5"><?php echo $completion_percentage; ?>% <?php echo ($project['status'] === 'Completed') ? 'Complete' : 'Timeline'; ?></p></div>
                                <?php endif; ?>
                                <p class="infographic-caption mt-3 text-2xl text-gray-200 leading-relaxed"></p>
                                <p class="hashtag-container text-right text-base font-bold mt-3 text-brand-orange/80">#GladysWangaDelivers</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- ====== CONTROLS PANE ====== -->
        <div class="lg:col-span-1">
            <h2 class="text-xl font-bold mb-4">Customize Infographic</h2>
            <div class="bg-white p-6 rounded-lg shadow space-y-6">
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">1. Select Infographic Type</label>
                    <fieldset class="mt-2 grid grid-cols-3 gap-4">
                        <div class="flex items-center"><input id="type-classic" name="infographic-type" type="radio" value="classic" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple" checked><label for="type-classic" class="ml-2 block text-sm font-medium text-gray-700">Classic</label></div>
                        <div class="flex items-center"><input id="type-modern" name="infographic-type" type="radio" value="modern" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple"><label for="type-modern" class="ml-2 block text-sm font-medium text-gray-700">Modern</label></div>
                        <div class="flex items-center"><input id="type-info" name="infographic-type" type="radio" value="project-info" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple"><label for="type-info" class="ml-2 block text-sm font-medium text-gray-700">Info Card</label></div>
                    </fieldset>
                </div>
                
                <hr/>

                <!-- Control Group: Modern Templates -->
                <div class="control-group" data-mode="modern project-info">
                    <label class="block text-sm font-medium text-gray-700 mb-2">2. Choose a Template</label>
                    <fieldset id="template-selector" class="mt-2 grid grid-cols-3 gap-x-4 gap-y-2"></fieldset>
                </div>
                
                <!-- Control Group: Image Selector (label changes) -->
                <div>
                    <label id="image-selector-label" class="block text-sm font-medium text-gray-700 mb-2">3. Select Image</label>
                    <div id="thumbnail-selector" class="grid grid-cols-4 gap-2 h-40 overflow-y-auto bg-gray-100 p-2 rounded-md">
                        <?php if(empty($all_photos)): ?>
                            <p class="col-span-full text-sm text-gray-500 text-center my-auto">No photos available for this project.</p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Common Controls -->
                <div id="layout-controls">
                    <label class="block text-sm font-medium text-gray-700 mb-2">4. Choose a Layout</label>
                     <fieldset class="mt-2"><div class="flex items-center space-x-4">
                        <div class="flex items-center"><input id="layout-banner-bottom" name="layout" type="radio" value="layout-banner-bottom" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple" checked><label for="layout-banner-bottom" class="ml-2 block text-sm font-medium text-gray-700">Banner Bottom</label></div>
                        <div class="flex items-center"><input id="layout-classic-overlay" name="layout" type="radio" value="layout-classic-overlay" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple"><label for="layout-classic-overlay" class="ml-2 block text-sm font-medium text-gray-700">Overlay</label></div>
                    </div></fieldset>
                </div>
                 <div id="element-toggles">
                    <label class="block text-sm font-medium text-gray-700 mb-2">5. Toggle Elements</label>
                    <div class="space-y-2">
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-hbc-logo" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-hbc-logo" class="font-medium text-gray-700">Show Homa Bay Logo</label></div></div>
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-genowa-logo" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-genowa-logo" class="font-medium text-gray-700">Show Genowa Logo</label></div></div>
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-hashtag" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-hashtag" class="font-medium text-gray-700">Show Hashtag</label></div></div>
                    </div>
                </div>
                <div id="caption-controls">
                    <label for="caption-input" class="block text-sm font-medium text-gray-700 mb-2">6. Enter a Caption</label>
                    <textarea id="caption-input" rows="4" maxlength="250" class="w-full rounded-md border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 focus:ring-brand-purple focus:ring-opacity-50" placeholder="Type a brief caption..."></textarea>
                    <div id="char-counter" class="text-right text-sm text-gray-500 mt-1">0 / 250</div>
                </div>
                <div>
                    <button id="download-btn" class="w-full inline-flex justify-center items-center rounded-md border border-transparent bg-brand-orange px-6 py-3 text-base font-medium text-white shadow-sm hover:bg-opacity-90 disabled:bg-gray-400">
                        Download as PNG
                    </button>
                </div>
            </div>
        </div>
    </div>
    <?php elseif ($project_id): ?>
        <div class="bg-red-100 text-red-700 p-4 rounded-lg">Project with ID <?php echo e($project_id); ?> not found or not approved.</div>
    <?php endif; ?>

</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    new TomSelect('#project-select',{ create: false, sortField: { field: "text", direction: "asc" } });
    
    <?php if ($project): ?>
        // --- DOM ELEMENT REFERENCES ---
        const container = document.getElementById('infographic-container');
        const contentWrapper = document.querySelector('.content-wrapper');
        const previewWrapper = document.getElementById('preview-wrapper');
        const templateLayer = document.getElementById('template-layer');
        const photoLayer = document.getElementById('photo-layer');
        const gradientLayer = document.getElementById('gradient-layer');
        const thumbnailSelector = document.getElementById('thumbnail-selector');
        const templateSelector = document.getElementById('template-selector');
        const imageSelectorLabel = document.getElementById('image-selector-label');
        const captionControls = document.getElementById('caption-controls');
        const captionInput = document.getElementById('caption-input');
        const downloadBtn = document.getElementById('download-btn');
        const elementToggles = document.getElementById('element-toggles');
        const allWardNameElements = document.querySelectorAll('.ward-name');
        
        // --- DATA AND STATE ---
        const allPhotos = <?php echo json_encode($all_photos); ?>;
        const projectStatus = '<?php echo $project['status']; ?>'.toLowerCase();
        const maxChars = 250;
        let activeMode = 'classic';
        let selectedPhotoUrl = allPhotos.length > 0 ? `uploads/${allPhotos[0].image_url}` : '';
        let selectedTemplate;

        const templates = {
            ongoing: [
                { name: 'White', path: 'assets/templates/ongoing-white.png', colors: { main: '#333333', accent: '#FDB813' } },
                { name: 'Dark', path: 'assets/templates/ongoing-dark.png', colors: { main: '#FFFFFF', accent: '#FDB813' } },
                { name: 'Cream', path: 'assets/templates/ongoing-cream.png', colors: { main: '#333333', accent: '#FDB813' } }
            ],
            completed: [
                { name: 'White', path: 'assets/templates/completed-white.png', colors: { main: '#333333', accent: '#FDB813' } },
                { name: 'Dark', path: 'assets/templates/completed-dark.png', colors: { main: '#FFFFFF', accent: '#FDB813' } },
                { name: 'Cream', path: 'assets/templates/completed-cream.png', colors: { main: '#333333', accent: '#FDB813' } }
            ]
        };
        
        // --- CORE FUNCTIONS ---
        function updatePreview() {
            // Reset layers
            templateLayer.style.backgroundImage = 'none';
            photoLayer.style.backgroundImage = 'none';
            photoLayer.classList.remove('photo-in-template');
            gradientLayer.style.opacity = '0';

            const isTemplateMode = activeMode === 'modern' || activeMode === 'project-info';

            if (isTemplateMode && selectedTemplate) {
                templateLayer.style.backgroundImage = `url(${selectedTemplate.path})`;
                photoLayer.style.backgroundImage = `url(${selectedPhotoUrl})`;
                photoLayer.classList.add('photo-in-template');
                contentWrapper.style.color = selectedTemplate.colors.main;
                allWardNameElements.forEach(el => el.style.color = selectedTemplate.colors.accent);
            } else { // Classic mode
                photoLayer.style.backgroundImage = `url(${selectedPhotoUrl})`;
                gradientLayer.style.opacity = '1';
                contentWrapper.style.color = '#FFFFFF';
                allWardNameElements.forEach(el => el.style.color = ''); // Reset to inherit from parent
            }
        }

        function switchMode(mode) {
            activeMode = mode;
            container.classList.remove('mode-modern', 'mode-project-info');
            if (mode === 'modern' || mode === 'project-info') {
                container.classList.add(`mode-${mode}`);
            }
            
            // Show/hide and enable/disable relevant controls
            document.querySelectorAll('.control-group').forEach(el => {
                el.style.display = el.dataset.mode.includes(mode) ? 'block' : 'none';
            });
            
            const isClassic = mode === 'classic';
            elementToggles.style.display = isClassic ? 'block' : 'none';
            captionControls.style.display = mode !== 'project-info' ? 'block' : 'none';
            
            imageSelectorLabel.textContent = isClassic ? '2. Select Background Image' : '3. Select Project Image';
            updatePreview();
            validateForDownload();
        }

        function validateForDownload() {
            const isCaptionRequired = activeMode === 'modern';
            const isCaptionValid = !isCaptionRequired || (captionInput.value.trim() !== '' && captionInput.value.length <= maxChars);
            const isPhotoSelected = selectedPhotoUrl !== '';
            const isTemplateSelected = activeMode === 'classic' || (activeMode !== 'classic' && selectedTemplate);
            downloadBtn.disabled = !(isCaptionValid && isPhotoSelected && isTemplateSelected);
        }

        function updatePreviewScale() {
            const scale = previewWrapper.offsetWidth / 1080;
            container.style.transform = `scale(${scale})`;
        }

        // --- EVENT LISTENERS ---

        // Infographic Type
        document.querySelectorAll('input[name="infographic-type"]').forEach(radio => {
            radio.addEventListener('change', (e) => switchMode(e.target.value));
        });

        // Layout
        document.querySelectorAll('input[name="layout"]').forEach(radio => {
            radio.addEventListener('change', (e) => {
                container.classList.remove('layout-banner-bottom', 'layout-classic-overlay');
                container.classList.add(e.target.value);
            });
        });

        // Toggles
        document.getElementById('toggle-hbc-logo').addEventListener('change', (e) => {
            document.querySelectorAll('.hbc-logo-wrapper').forEach(el => el.style.display = e.target.checked ? '' : 'none');
            updateLogoDivider();
        });
        document.getElementById('toggle-genowa-logo').addEventListener('change', (e) => {
            document.querySelectorAll('.genowa-logo-wrapper').forEach(el => el.style.display = e.target.checked ? '' : 'none');
            updateLogoDivider();
        });
        document.getElementById('toggle-hashtag').addEventListener('change', (e) => {
            document.querySelectorAll('.hashtag-container').forEach(el => el.style.display = e.target.checked ? '' : 'none');
        });
        function updateLogoDivider() {
            const hbcVisible = document.getElementById('toggle-hbc-logo').checked;
            const genowaVisible = document.getElementById('toggle-genowa-logo').checked;
            document.querySelectorAll('.logo-divider').forEach(el => el.style.display = (hbcVisible && genowaVisible) ? '' : 'none');
        }

        // Caption
        const charCounter = document.getElementById('char-counter');
        captionInput.addEventListener('input', () => {
             const currentLength = captionInput.value.length;
             charCounter.textContent = `${currentLength} / ${maxChars}`;
             charCounter.classList.toggle('text-red-600', currentLength > maxChars);
             document.querySelectorAll('.infographic-caption').forEach(el => el.textContent = captionInput.value);
             validateForDownload();
        });

        // Download Button
        downloadBtn.addEventListener('click', function() {
            downloadBtn.textContent = 'Generating...';
            downloadBtn.disabled = true;
            container.style.transform = 'scale(1)'; // Force 1:1 for capture

            const options = {
                useCORS: true,
                allowTaint: true,
                width: 1080,
                height: 1350,
                scale: activeMode === 'classic' ? 2 : 1 
            };

            setTimeout(() => {
                html2canvas(container, options)
                .then(canvas => {
                    const link = document.createElement('a');
                    link.download = `genowa-project-<?php echo preg_replace('/[^a-z0-9_]/i', '_', $project['name']); ?>.png`;
                    link.href = canvas.toDataURL('image/png');
                    link.click();
                })
                .catch(error => console.error('Error generating canvas:', error))
                .finally(() => {
                    downloadBtn.textContent = 'Download as PNG';
                    validateForDownload();
                    updatePreviewScale();
                });
            }, 500);
        });

        // Window Resize
        window.addEventListener('resize', updatePreviewScale);

        // --- INITIALIZATION ---

        // Populate Photo Thumbnails
        if (allPhotos.length > 0) {
            allPhotos.forEach((photo, index) => {
                const thumb = document.createElement('img');
                thumb.src = `uploads/${photo.image_url}`;
                thumb.className = "w-full h-auto object-cover rounded-md cursor-pointer border-2 border-transparent hover:border-brand-purple";
                if (index === 0) thumb.classList.add('selected');
                
                thumb.addEventListener('click', () => {
                    selectedPhotoUrl = `uploads/${photo.image_url}`;
                    document.querySelectorAll('#thumbnail-selector img').forEach(i => i.classList.remove('selected'));
                    thumb.classList.add('selected');
                    updatePreview();
                    validateForDownload();
                });
                thumbnailSelector.appendChild(thumb);
            });
        }

        // Populate Template Selector
        const availableTemplates = templates[projectStatus] || [];
        if(availableTemplates.length > 0) {
            selectedTemplate = availableTemplates[0]; // Set default
            availableTemplates.forEach((template, index) => {
                const id = `template-${index}`;
                let html = `<div class="flex items-center"><input id="${id}" name="template-choice" type="radio" value="${index}" class="h-4 w-4" ${index === 0 ? 'checked' : ''}><label for="${id}" class="ml-2 block text-sm">${template.name}</label></div>`;
                templateSelector.insertAdjacentHTML('beforeend', html);
            });
            document.querySelectorAll('input[name="template-choice"]').forEach(radio => {
                radio.addEventListener('change', (e) => {
                    selectedTemplate = availableTemplates[e.target.value];
                    updatePreview();
                });
            });
        }

        // Final Setup
        switchMode(activeMode); // Set initial control visibility
        updatePreview();
        validateForDownload();
        updatePreviewScale();
        updateLogoDivider();

    <?php endif; ?>
});
</script>

<?php include 'includes/footer.php'; ?>
