<?php
// edit_project.php

require_once 'db.php';
require_once 'functions.php';

require_login();

// Determine if we are editing or adding a new project
$is_editing = isset($_GET['id']) && is_numeric($_GET['id']);
$project = null;

if ($is_editing) {
    $project = get_project_by_id($db, $_GET['id']);
    // If project not found, redirect to the main page
    if (!$project) {
        header('Location: manage_projects.php');
        exit();
    }
}

// Generate a CSRF token for form submission
$csrf_token = generate_csrf_token();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $is_editing ? 'Edit Project' : 'Add New Project' ?></title>
    <script src="[https://cdn.tailwindcss.com](https://cdn.tailwindcss.com)"></script>
    <link rel="stylesheet" href="[https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css](https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css)">
</head>
<body class="bg-slate-100 font-sans">

    <div class="container mx-auto p-4 sm:p-6 lg:p-8">
        
        <header class="mb-8">
            <a href="manage_projects.php" class="text-blue-600 hover:underline mb-4 block"><i class="fas fa-arrow-left mr-2"></i>Back to Projects</a>
            <h1 class="text-3xl font-bold text-slate-800"><?= $is_editing ? 'Edit Project' : 'Create a New Project' ?></h1>
            <p class="text-slate-600 mt-1">Fill in the details below.</p>
        </header>

        <div class="bg-white rounded-xl shadow-xl p-6 md:p-8">
            <form id="project-form" enctype="multipart/form-data">
                <!-- Hidden fields for action, project ID, and CSRF token -->
                <input type="hidden" name="action" value="<?= $is_editing ? 'update' : 'create' ?>">
                <input type="hidden" name="id" value="<?= $is_editing ? $project['id'] : '' ?>">
                <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    
                    <!-- Project Name -->
                    <div class="col-span-1">
                        <label for="name" class="block text-sm font-medium text-slate-700">Project Name</label>
                        <input type="text" name="name" id="name" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['name'] ?? '') ?>">
                    </div>

                    <!-- Status -->
                    <div class="col-span-1">
                        <label for="status" class="block text-sm font-medium text-slate-700">Status</label>
                        <select id="status" name="status" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500">
                            <option value="Planned" <?= (($project['status'] ?? '') == 'Planned') ? 'selected' : '' ?>>Planned</option>
                            <option value="Ongoing" <?= (($project['status'] ?? '') == 'Ongoing') ? 'selected' : '' ?>>Ongoing</option>
                            <option value="Completed" <?= (($project['status'] ?? '') == 'Completed') ? 'selected' : '' ?>>Completed</option>
                            <option value="Stalled" <?= (($project['status'] ?? '') == 'Stalled') ? 'selected' : '' ?>>Stalled</option>
                        </select>
                    </div>

                    <!-- Description -->
                    <div class="md:col-span-2">
                        <label for="description" class="block text-sm font-medium text-slate-700">Description</label>
                        <textarea id="description" name="description" rows="4" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"><?= htmlspecialchars($project['description'] ?? '') ?></textarea>
                    </div>

                    <!-- Budget Allocated -->
                    <div class="col-span-1">
                        <label for="budget_allocated" class="block text-sm font-medium text-slate-700">Budget Allocated ($)</label>
                        <input type="number" name="budget_allocated" id="budget_allocated" step="0.01" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['budget_allocated'] ?? '') ?>">
                    </div>
                    
                    <!-- Contractor Name -->
                    <div class="col-span-1">
                        <label for="contractor_name" class="block text-sm font-medium text-slate-700">Contractor Name</label>
                        <input type="text" name="contractor_name" id="contractor_name" class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['contractor_name'] ?? '') ?>">
                    </div>

                    <!-- Start Date -->
                    <div class="col-span-1">
                        <label for="start_date" class="block text-sm font-medium text-slate-700">Start Date</label>
                        <input type="date" name="start_date" id="start_date" class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['start_date'] ?? '') ?>">
                    </div>

                    <!-- Expected Completion Date -->
                    <div class="col-span-1">
                        <label for="expected_completion_date" class="block text-sm font-medium text-slate-700">Expected Completion Date</label>
                        <input type="date" name="expected_completion_date" id="expected_completion_date" class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['expected_completion_date'] ?? '') ?>">
                    </div>
                    
                    <!-- Ward ID & Manifesto Pillar ID (using placeholders) -->
                    <div class="col-span-1">
                        <label for="ward_id" class="block text-sm font-medium text-slate-700">Ward ID</label>
                        <input type="number" name="ward_id" id="ward_id" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['ward_id'] ?? '1') ?>">
                    </div>
                     <div class="col-span-1">
                        <label for="manifesto_pillar_id" class="block text-sm font-medium text-slate-700">Manifesto Pillar ID</label>
                        <input type="number" name="manifesto_pillar_id" id="manifesto_pillar_id" required class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-blue-500 focus:ring-blue-500" value="<?= htmlspecialchars($project['manifesto_pillar_id'] ?? '1') ?>">
                    </div>

                    <!-- Project Image Upload -->
                    <div class="md:col-span-2">
                        <label class="block text-sm font-medium text-slate-700">Project Image</label>
                        <div class="mt-2 flex items-center space-x-6">
                            <div id="image-preview" class="shrink-0">
                                <?php if ($is_editing && !empty($project['project_image'])): ?>
                                    <img class="h-20 w-20 object-cover rounded-md" src="uploads/<?= htmlspecialchars($project['project_image']) ?>" alt="Current project image">
                                <?php else: ?>
                                    <span class="h-20 w-20 rounded-md bg-slate-200 flex items-center justify-center text-slate-500">
                                        <i class="fas fa-image fa-2x"></i>
                                    </span>
                                <?php endif; ?>
                            </div>
                            <label class="block">
                                <span class="sr-only">Choose profile photo</span>
                                <input type="file" name="project_image" id="project_image" accept="image/png, image/jpeg, image/gif" class="block w-full text-sm text-slate-500
                                  file:mr-4 file:py-2 file:px-4
                                  file:rounded-full file:border-0
                                  file:text-sm file:font-semibold
                                  file:bg-blue-50 file:text-blue-700
                                  hover:file:bg-blue-100
                                "/>
                            </label>
                        </div>
                         <p class="text-xs text-slate-500 mt-1">Accepted formats: JPG, PNG, GIF. Max size: 2MB.</p>
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="mt-8 pt-5 border-t border-slate-200">
                    <div class="flex justify-end space-x-4">
                        <a href="manage_projects.php" class="px-4 py-2 bg-slate-200 text-slate-800 rounded-lg hover:bg-slate-300">Cancel</a>
                        <button type="submit" id="submit-btn" class="px-6 py-2 bg-blue-600 text-white font-bold rounded-lg hover:bg-blue-700 disabled:bg-blue-300">
                            <i class="fas fa-save mr-2"></i>
                            <?= $is_editing ? 'Save Changes' : 'Create Project' ?>
                        </button>
                    </div>
                </div>
            </form>
            <!-- Submission Status Message -->
            <div id="status-message" class="mt-4 text-center"></div>
        </div>
    </div>

    <script>
        const form = document.getElementById('project-form');
        const submitBtn = document.getElementById('submit-btn');
        const statusMessage = document.getElementById('status-message');
        const imageInput = document.getElementById('project_image');
        const imagePreview = document.getElementById('image-preview');

        // Image Preview Logic
        imageInput.addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                const reader = new FileReader();
                imagePreview.innerHTML = ''; // Clear previous preview
                reader.onload = function(e) {
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.className = 'h-20 w-20 object-cover rounded-md';
                    imagePreview.appendChild(img);
                }
                reader.readAsDataURL(file);
            }
        });

        // Form Submission Logic
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Saving...';
            statusMessage.innerHTML = '';

            const formData = new FormData(this);

            fetch('api.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    statusMessage.innerHTML = `<div class="p-4 bg-green-100 text-green-800 rounded-lg">${data.message}</div>`;
                    // Redirect back to the main management page after a short delay
                    setTimeout(() => {
                        window.location.href = 'manage_projects.php';
                    }, 1500);
                } else {
                    statusMessage.innerHTML = `<div class="p-4 bg-red-100 text-red-800 rounded-lg">Error: ${data.message}</div>`;
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = '<i class="fas fa-save mr-2"></i>' + ('<?= $is_editing ?>' ? 'Save Changes' : 'Create Project');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                statusMessage.innerHTML = `<div class="p-4 bg-red-100 text-red-800 rounded-lg">An unexpected network error occurred.</div>`;
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-save mr-2"></i>' + ('<?= $is_editing ?>' ? 'Save Changes' : 'Create Project');
            });
        });
    </script>
</body>
</html>
