<?php
// FILE: admin/audit_log.php
// VERSION 1.1 - Added a 'Details' column to show specific change information.

// SETS THE DEFAULT TIMEZONE FOR THE ENTIRE APPLICATION
date_default_timezone_set('Africa/Nairobi');

require_once 'functions.php';
require_once 'db.php';

// Security: Only SuperAdmins can access this page
require_login();
require_super_admin();

// --- Pagination Setup ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 25; // Number of log entries per page
$offset = ($page - 1) * $limit;

$audit_logs = [];
$total_logs = 0;

try {
    // Get total count for pagination
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM audit_trails");
    $total_logs = $count_stmt->fetchColumn();
    $total_pages = ceil($total_logs / $limit);

    // Fetch the logs for the current page
    $log_stmt = $pdo->prepare("SELECT * FROM audit_trails ORDER BY timestamp DESC LIMIT :limit OFFSET :offset");
    $log_stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $log_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $log_stmt->execute();
    $audit_logs = $log_stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching audit trails: " . $e->getMessage();
}

include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">System Audit Trails</h1>
        <p class="mt-2 text-sm text-gray-700">A record of all significant actions performed by users in the system.</p>
    </div>
</header>
    
<main class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-8">
    <div class="overflow-x-auto bg-white rounded-lg shadow">
        <?php if (isset($db_error)): ?>
            <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
        <?php elseif (empty($audit_logs)): ?>
            <div class="text-center p-12">
                <h3 class="mt-2 text-lg font-medium text-gray-900">No system actions have been logged yet.</h3>
            </div>
        <?php else: ?>
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Timestamp</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Target</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Details</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($audit_logs as $log): ?>
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900"><?php echo date('M j, Y, g:i a', strtotime($log['timestamp'])); ?></div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm font-medium text-gray-900"><?php echo e($log['user_name']); ?></div>
                                <div class="text-sm text-gray-500">ID: <?php echo e($log['user_id']); ?></div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                    <?php echo e($log['action']); ?>
                                </span>
                            </td>
                             <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900"><?php echo e($log['target_type']); ?> #<?php echo e($log['target_id']); ?></div>
                            </td>
                            <td class="px-6 py-4">
                                <div class="text-sm text-gray-700"><?php echo e($log['details'] ?? 'N/A'); ?></div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <div class="mt-6 flex justify-center items-center">
        <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>" class="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">Prev</a>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                <a href="?page=<?php echo $i; ?>" class="relative inline-flex items-center px-4 py-2 border border-gray-300 <?php echo $i == $page ? 'z-10 bg-brand-light-gray text-brand-purple' : 'bg-white text-gray-700 hover:bg-gray-50'; ?> text-sm font-medium">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>

            <?php if ($page < $total_pages): ?>
                <a href="?page=<?php echo $page + 1; ?>" class="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">Next</a>
            <?php endif; ?>
        </nav>
    </div>
</main>

<?php include 'includes/footer.php'; ?>