// FILE: admin/assets/js/main.js

document.addEventListener('DOMContentLoaded', function () {
    const htmlEl = document.documentElement;
    const themeToggleButtons = document.querySelectorAll('.theme-toggle-btn');
    
    // --- THEME LOGIC ---
    function updateAllToggleIcons() {
        const isDark = htmlEl.classList.contains('dark');
        themeToggleButtons.forEach(btn => {
            const darkIcon = btn.querySelector('.theme-toggle-dark-icon');
            const lightIcon = btn.querySelector('.theme-toggle-light-icon');
            if (darkIcon && lightIcon) {
                lightIcon.classList.toggle('hidden', !isDark);
                darkIcon.classList.toggle('hidden', isDark);
            }
        });
    }

    themeToggleButtons.forEach(btn => {
        btn.addEventListener('click', () => {
            htmlEl.classList.toggle('dark');
            const newTheme = htmlEl.classList.contains('dark') ? 'dark' : 'light';
            localStorage.setItem('theme', newTheme);
            updateAllToggleIcons();
        });
    });
    
    // Initialize icons on load
    updateAllToggleIcons();

    // --- HEADER SCROLL EFFECT ---
    const header = document.querySelector('header');
    if (header) {
        window.addEventListener('scroll', () => {
            // Toggles a shadow class if scrolled more than 10px
            header.classList.toggle('shadow-md', window.scrollY > 10);
        }, { passive: true });
    }

    // --- MOBILE MENU ---
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const mobileMenu = document.getElementById('mobile-menu');
    if (mobileMenuButton && mobileMenu) {
        const menuOpenIcon = document.getElementById('menu-open-icon');
        const menuCloseIcon = document.getElementById('menu-close-icon');
        
        mobileMenuButton.addEventListener('click', function () {
            const isExpanded = this.getAttribute('aria-expanded') === 'true';
            this.setAttribute('aria-expanded', !isExpanded);
            mobileMenu.classList.toggle('hidden');
            menuOpenIcon.classList.toggle('hidden');
            menuCloseIcon.classList.toggle('hidden');
        });
    }

    // --- LOCATION MODAL LOGIC ---
    const locationModal = document.getElementById('location-modal');
    const locationSelector = document.getElementById('location-selector');
    let lastFocused;

    // Make these global if needed by inline onclicks, otherwise keep scoped
    window.openLocationModal = function() {
        if (!locationModal) return;
        lastFocused = document.activeElement;
        locationModal.classList.remove('hidden');
        // Small timeout to allow display change before focus
        setTimeout(() => locationSelector.focus(), 100);
        trapFocus(locationModal);
    };

    function closeModal() {
        if (!locationModal) return;
        locationModal.classList.add('hidden');
        if (lastFocused) lastFocused.focus();
    }

    // Bind Open Buttons
    const openBtns = document.querySelectorAll('#set-location-btn, #set-location-btn-mobile');
    openBtns.forEach(btn => btn.addEventListener('click', window.openLocationModal));

    // Update Badges UI
    function updateConstituencyUI(name) {
        document.querySelectorAll('.my-constituency-badge').forEach(badge => {
            badge.textContent = name;
            badge.classList.toggle('hidden', !name);
        });
    }

    // Save Button Logic
    const saveBtn = document.getElementById('save-location-btn');
    if (saveBtn) {
        const saveBtnSpinner = saveBtn.querySelector('svg');
        const saveBtnText = saveBtn.querySelector('.button-text');
        
        saveBtn.addEventListener('click', () => {
            const selectedId = locationSelector.value;
            const selectedName = locationSelector.options[locationSelector.selectedIndex].text;
            
            if (!selectedId) { 
                alert('Please select a constituency.'); 
                return; 
            }
            
            saveBtn.disabled = true;
            saveBtnSpinner.classList.remove('hidden');
            saveBtnText.textContent = 'Saving...';
            
            fetch('/api/save_constituency.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                body: JSON.stringify({ constituency_id: selectedId })
            })
            .then(res => res.ok ? res.json() : Promise.reject(res))
            .then(data => {
                if (data.status === 'success') {
                    updateConstituencyUI(selectedName);
                    const announcer = document.getElementById('aria-announcer');
                    if(announcer) announcer.textContent = `Constituency updated to ${selectedName}.`;
                    closeModal();
                } else { 
                    throw new Error(data.message || 'Failed to save choice.'); 
                }
            })
            .catch(err => {
                console.error('Save failed:', err);
                alert('An error occurred. Your choice could not be saved.');
            })
            .finally(() => {
                saveBtn.disabled = false;
                saveBtnSpinner.classList.add('hidden');
                saveBtnText.textContent = 'Save My Choice';
            });
        });
    }

    // Clear Button Logic
    const clearBtn = document.getElementById('clear-location-btn');
    if (clearBtn) {
        clearBtn.addEventListener('click', () => {
            clearBtn.disabled = true;
            fetch('/api/save_constituency.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                body: JSON.stringify({ constituency_id: null })
            })
            .then(res => res.ok ? res.json() : Promise.reject(res))
            .then(data => {
                if (data.status === 'success') {
                    updateConstituencyUI('');
                    if(locationSelector) locationSelector.value = '';
                    const announcer = document.getElementById('aria-announcer');
                    if(announcer) announcer.textContent = 'Constituency selection cleared.';
                    closeModal();
                } else { 
                    throw new Error(data.message || 'Failed to clear choice.'); 
                }
            })
            .catch(err => {
                console.error('Clear failed:', err);
                alert('An error occurred. Your choice could not be cleared.');
            })
            .finally(() => { clearBtn.disabled = false; });
        });
    }

    // Modal Accessibility (Close on click outside or Escape)
    if(locationModal){
        locationModal.addEventListener('click', (e) => { if (e.target === locationModal) closeModal(); });
        locationModal.addEventListener('keydown', (e) => { if (e.key === "Escape") closeModal(); });
    }

    // Focus Trap
    function trapFocus(element) {
        const focusableEls = element.querySelectorAll('button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])');
        if (focusableEls.length === 0) return;
        
        const firstEl = focusableEls[0]; 
        const lastEl = focusableEls[focusableEls.length - 1];
        
        element.addEventListener('keydown', function(e) {
            if (e.key !== 'Tab') return;
            if (e.shiftKey) { 
                if (document.activeElement === firstEl) { 
                    e.preventDefault(); 
                    lastEl.focus(); 
                }
            } else { 
                if (document.activeElement === lastEl) { 
                    e.preventDefault(); 
                    firstEl.focus(); 
                } 
            }
        });
    }
});