<?php
// api.php

// Set header to return JSON
header('Content-Type: application/json');

// Include core files
require_once 'db.php';
require_once 'functions.php';

// Secure this endpoint by requiring a login
require_login();

// Basic request validation
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
    exit();
}

// --- CSRF Protection ---
if (!isset($_POST['csrf_token']) || !verify_csrf_token($_POST['csrf_token'])) {
    echo json_encode(['success' => false, 'message' => 'CSRF token validation failed. Please refresh and try again.']);
    exit();
}

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'create':
    case 'update':
        handle_create_update($db, $action);
        break;
    
    case 'delete':
        handle_delete($db);
        break;

    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action specified.']);
        break;
}

/**
 * Handles the logic for creating or updating a project.
 */
function handle_create_update($db, $action) {
    // --- Data Validation ---
    $required_fields = ['name', 'description', 'status', 'budget_allocated', 'ward_id', 'manifesto_pillar_id'];
    foreach ($required_fields as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Field '{$field}' is required."]);
            exit();
        }
    }

    // --- Image Handling ---
    $image_filename = $_POST['current_image'] ?? null; // For updates
    if (isset($_FILES['project_image']) && $_FILES['project_image']['error'] == UPLOAD_ERR_OK) {
        $new_filename = handle_image_upload($_FILES['project_image']);
        if ($new_filename) {
            // If a new image is uploaded, delete the old one during an update
            if ($action === 'update' && !empty($image_filename)) {
                 $old_path = 'uploads/' . $image_filename;
                 if(file_exists($old_path)) unlink($old_path);
            }
            $image_filename = $new_filename;
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid image file. Please upload a valid JPG, PNG, or GIF.']);
            exit();
        }
    }
    
    // Prepare data array for database function
    $data = [
        'name' => trim($_POST['name']),
        'description' => trim($_POST['description']),
        'status' => $_POST['status'],
        'budget_allocated' => $_POST['budget_allocated'],
        'start_date' => $_POST['start_date'],
        'expected_completion_date' => $_POST['expected_completion_date'],
        'contractor_name' => trim($_POST['contractor_name']),
        'ward_id' => $_POST['ward_id'],
        'manifesto_pillar_id' => $_POST['manifesto_pillar_id'],
        'project_image' => $image_filename
    ];

    if ($action === 'update') {
        if (empty($_POST['id'])) {
            echo json_encode(['success' => false, 'message' => 'Project ID is missing for update.']);
            exit();
        }
        $data['id'] = $_POST['id'];
        $result = update_project($db, $data);
        $message = 'Project updated successfully!';
    } else {
        $result = create_project($db, $data);
        $message = 'Project created successfully!';
    }

    if ($result) {
        echo json_encode(['success' => true, 'message' => $message]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Database operation failed.']);
    }
}

/**
 * Handles the logic for deleting a project.
 */
function handle_delete($db) {
    $id = $_POST['id'] ?? null;
    if (empty($id)) {
        echo json_encode(['success' => false, 'message' => 'Project ID is required for deletion.']);
        exit();
    }

    if (delete_project($db, $id)) {
        echo json_encode(['success' => true, 'message' => 'Project deleted successfully.']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to delete the project.']);
    }
}
?>
